<?php

declare(strict_types=1);

namespace XPayrPlugin;

use XPayrPlugin\Service\XPayrPayment;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Plugin;
use Shopware\Core\Framework\Plugin\Context\ActivateContext;
use Shopware\Core\Framework\Plugin\Context\DeactivateContext;
use Shopware\Core\Framework\Plugin\Context\InstallContext;
use Shopware\Core\Framework\Plugin\Context\UninstallContext;
use Shopware\Core\Framework\Plugin\Util\PluginIdProvider;
use Shopware\Core\System\SystemConfig\SystemConfigService;

/**
 * XPayr Crypto Payment Gateway — Shopware 6 Plugin
 *
 * Registers the XPayr asynchronous payment method and manages
 * its lifecycle (install, activate, deactivate, uninstall).
 */
class XPayrPlugin extends Plugin
{
    /**
     * Register the XPayr payment method and set webhook URL on install.
     */
    public function install(InstallContext $context): void
    {
        $this->addPaymentMethod($context->getContext());
        $this->setWebhookUrl();
    }

    /**
     * Deactivate the payment method on uninstall (preserving order data).
     */
    public function uninstall(UninstallContext $context): void
    {
        $this->setPaymentMethodIsActive(false, $context->getContext());
    }

    /**
     * Activate the payment method and refresh webhook URL when activated.
     */
    public function activate(ActivateContext $context): void
    {
        $this->addPaymentMethod($context->getContext()); // Ensure it exists and is updated
        $this->setPaymentMethodIsActive(true, $context->getContext());
        $this->setWebhookUrl();
        parent::activate($context);
    }

    /**
     * Deactivate the payment method when the plugin is deactivated.
     */
    public function deactivate(DeactivateContext $context): void
    {
        $this->setPaymentMethodIsActive(false, $context->getContext());
        parent::deactivate($context);
    }

    /**
     * Auto-populate the webhook URL config value from APP_URL.
     */
    private function setWebhookUrl(): void
    {
        /** @var SystemConfigService $configService */
        $configService = $this->container->get(SystemConfigService::class);

        $appUrl = rtrim((string) ($_ENV['APP_URL'] ?? $_SERVER['APP_URL'] ?? ''), '/');

        if (empty($appUrl)) {
            $appUrl = rtrim((string) $configService->get('core.app.appUrl'), '/');
        }

        if (!empty($appUrl)) {
            $webhookUrl = $appUrl . '/xpayr/webhook';
            $configService->set('XPayrPlugin.config.webhookUrl', $webhookUrl);
        }
    }

    /**
     * Create or update the XPayr payment method entry in the database.
     */
    private function addPaymentMethod(Context $context): void
    {
        /** @var PluginIdProvider $pluginIdProvider */
        $pluginIdProvider = $this->container->get(PluginIdProvider::class);
        $pluginId = $pluginIdProvider->getPluginIdByBaseClass(get_class($this), $context);

        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');

        $paymentMethodExists = $this->getPaymentMethodId();

        // Try to find the default availability rule (e.g., "Cart >= 0" or "All customers")
        $ruleRepository = $this->container->get('rule.repository');
        $ruleCriteria = new Criteria();
        $ruleCriteria->addFilter(new \Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter('name', ['Cart >= 0', 'All customers', 'Always valid']));
        $ruleCriteria->setLimit(1);
        $ruleId = $ruleRepository->searchIds($ruleCriteria, $context)->firstId();

        // Fallback to any active rule if default is not found
        if ($ruleId === null) {
            $fallbackCriteria = new Criteria();
            $fallbackCriteria->addFilter(new \Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter('invalid', false));
            $fallbackCriteria->setLimit(1);
            $ruleId = $ruleRepository->searchIds($fallbackCriteria, $context)->firstId();
        }

        $paymentMethodData = [
            'handlerIdentifier' => XPayrPayment::class,
            'technicalName' => 'payment_xpayr_crypto',
            'name' => 'XPayr',
            'description' => '<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAApEAAAKRCAYAAAAMK/LLAAAACXBIWXMAACMQAAAjEAFnL93WAAAgAElEQVR4nOzd15Nc553m+e/vPZnlUfAESAp0IOhAb0ArdbN79mK00XMnxE5PtKI1E7H8MzD8M3CxkmK00xtRuFNHUBEbsw1FS/QgCHoPgiB8AShvM8/77MWpRBVBmIJUhXLPh1EEUGnqrcyT533Oa8HMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMllAsdQEAkIL/foWy/Hd06e8R+tHtrcde63ZbPVrvNXDF4+XVyDezOGZmfzXXYbaCLX2I7FMbY4NdtNfbiSgAaM6Ex3bNfqia/PADVpsp+1TEpfs2EU2JDm7sw9jU1e/fuMHnmq/Oa/xMgIlYuvfmamVrlak+57iZ+/rU53E8te7fum9tHr9nM9Klv9cIpiIuvf/FzOOnlGnPTcaaDbo3TAHT7I3yus9tZnaz9akA2mCgjcl6OwAdM/XXfOuc652n53PfhXSjdVarTEtZ111uPnXYXK3661oZYr7P0TL3uW40f9SJH9aXyjQQtZypTzfo2drgbRoL2eBSW6gn+ovsVx0mtlJPO4ooN6PUARQUM7c3EHHpPxBzX9xERFBTkAlKhJSpkWlW/5pXGZJ06bCZ+5jW39sue5485yBLM7fNfVye+Xu67GDMc+5TSJRArl29jG3NoJw5uC5/rrmCQIiMKOYcfGXEFR+Xr/C6tO7Xuq1ERG32ICsJUjMoqJ5TpJl3BNrIM4+NmdLM/kzNvH9z/1279JpV927MPFdcVtbWaxoE0uztJUFSkIGIoCmQRKKk1FTZXhuiOXIW5bNIg766N7NlpU8FEyMbqLMV1W8pknopSDQunVN/XA+1hGbOtzVd8VwO1YV2asal+iU1g0ZNFAvYIFK2LuLnPGftGvVUas7e1ipXY+bfbTO3teq7Vj1WziNcXq1ubNUX18oBl9c58MP6/XqCqv6ZTSfXf32v9DOblz1MlzKEqM/8/fL6fa5WXV89d0GdYqY0AjJ1yqA22cz1QQYmz7Fz6DzS+ELVjUt3BdCngubIxhrpiRz6KeJeET1IXTMvRqZ6ORNBQRU7Wr90pCpz11BU4QUEygQZzTwWlGcfc+mxc/7OZbfN/c6Pvzd7AKQf3fbjn3N1VbiaeyVw5fehFW+DOb/jDx4TCWLmd8w/KHOQrvK8AsorPFe6VLbqO7rsUXPLkub83EsSl8rZKusPbs8//J5mHlNc6TFzn/uy5708qGaq97kkYizQMbI+ytKH5HSUHT0XeTmaV3gdzMxurv2qs25sE5HvS8QTwG6JW4iooTynXkmX1SM5Zr6XQbrUdPBjM+fyqwWwKwSRued6XaUuqu44e9uVwtDViKjKD7O/w5zn+MHPvPR7MvuYlrmvSZ59bKsh5cplu0aL2+Wv8WXPO1frZ7Reqx/Wh1evNy93zdf38sen2ef8UR5pvY8x837nqF6LSGimcTDIoJJIzcgaI8WxyHxUFjpM6jnG3pi4ZlnmaelaIrcSnCnalPNmxE6CJwJtJqJrpuVp5kCL+MFYuBkKaohiJmvMbUvMP7zbFUPk3H9f2eXtYj/8TvzoHld+vsu/1zrwfhSwrliCyw/jK4TIOQXTZQfwlcrYKtPlgXP2vlc7xH/4/fSDnzv7xJf/zB/8jvHDf2vmf3PD7qX3Kn78vLO/0+Vtw637S5PA3QTrUkr1jApOj33NQV1wkDSzJXVQNU63AmR6VuilEA8HbEAqfhj8rlR1tE6Zl87eVwuRAbpSQ8eVnvjyBpO47M859abmPuZarlBXRauhZ/bRunTfqzz0WlVsVP+7vLQ/drWS6so3ReuPy58xflBvX6l2vrqrBdzrlFNz/7hKw1erLSZaz1y1RlYyRImUCSZDul0RuSZONhtDZ4AVHiL/SOaecirqaVDKp4K4Q7ABVCOoz95R13rhW+Ye8Ff78NhcS9cGvZjagDpEE6lIRRRZOXN8pMk+DXjijZktiT4VnB7bXAs9mBUvCp4N4hHQrTC3vjP7K8WPLgAAugSDSL2kaKOZUtVY99d3aS9diHw1Mvs13Gwf+qpQahNMh5DgkZlyXa079lpWZzSy+UpAJ2IXKbrJ1AsYK9vTOHcNlkhDHiNpZjfVPiUYXk/Ezky8FMH/jnQvsJ6qrnO9ZYth7hAxAQ3QRDNrnObUNNG7IHXh0rbavRIN0vTZsskXIT6Q+BA4BTEJl8Y1ms1XNT4kaEdsI7hfxKNF5hHa6nfxL0MbqhO6mdlNsF917h/bShkPJKU9gqcl7sEB0hZfKzw2gSGhU5BOkOMiaUvjOo+dt6WdnQ3wi61j/Pr8qbKzs71IuUeZ3pnxkDuAdfhDZjeuapGEHRE8I6mtiKBMxRS7+xtIY26RNLNF1acCRjcwrXtT4meCF4N4CLSB5VD32lpQAiNBHAN9GsRXJC5wjOmF+gFLfyBHCGmUfxn6vkz1WpHKQlkQUacKAq2rNYdJm49WE34BrBc8QNAh8lhRFhdL1Sf4LU1gcmmLaWar2tjgOtra7khFflxZP43QU4hWgHR9ZjfDJHBCoQ8ip8NlUX5N77rBhZwfsDy69iLEbetHSM1jJXxCpA+EvgEGqFaT8oQIuxGti44C6AF2SDykKB+u5fq91MY3VmuUmpktMCn4vbqo124rKB9BeiYi7gM2Uk2iWR71rq1WmaoFcho4h/RhZL1Tig9Jk6f4jwvXCgnL6WB+OZrQO0pZO57gkyAOCb0PcZrqxZjPsjhmc7WC5LpI6QGRfpbR87Vavp91Y5tmdo0wM1sYUvBb2hkZvr2IeEKRfgbxFHALboG0xddah3MM+BZ4lxRvlBRHiOYJPtm6YIuMtyx9d/Zce2Oa3+tic3zkaCE6IUKijKqct+ClEOzGBVBH2h4oKVLOWQNQjsNwg31a0KZ9M1vDDlCnY2JrUcb9ivwUxOPAHUDXUhfN1owGcBLxPsHrmXSIZnmMzvXDi1HXLa8QCfAPTPDrqdNlZ3sqqlXZJWgLkQm2Uq0F+Jcs/2NrU6trux3F1gjdR2ggkXKezom7Bj/30j9m9lfbrzrlxLYi6xFFPKvgsagmiHbjOssWV2snukngLOJDgtez0tuocZSj64cWq7Fk+YXI1kSbX58/WXa35yKrJKIhmAjxOMF2llM3vK0UQdAG3ArxrGBDStGTO9umOdD/pWdsm9lfrE8FU+NbCpqPKtJ/CHgK2MnsCiMOkLZYWl3YE8CJSy2QSm9Tn/oaNo4uZm/b8guRMBskD/SfKumgyAJFewTrVa2vVeArO7txCehC7IigSxHNotTJko5Rfnf2JNU4EjOz+TuoGv1jW2o13Z+VngrYI9gF9OKJNLZ4Wo0eJWKa4DTwAYk3Mvk9GjrGP24cXuzGkeV7cEeIT7aO02ycKVN8HRGfSnwGnEZM4Yk2dmNmZ2wHHcCWEDtFfrwQuyk6ttGntiUuo5mtJPuUODG+tdbkkQwvidgjuAfYQDX0yg0dtpiawDjBCcEHBG9m8iGKdJQ71t2UYVrLN0RCtTXiV5tGSY1TpeJL4BPBxwQngFGqMQCeFGHzFXO+6sA2RTwmpWdqUbuPqfHNnrFtZvOyT4l7RjaS8q4cekHiZ0F+mCpAFrgb2xZPKxxOAqcEHwTxVkaHKPgKei5WK94svuUdIqEKkp9sHKGukynxcRBvIg4B31GNAXBrpN2oantEWB/wgNAzWflJinInjaFeB0kzu6ZWgCy0K0U8KeLpgIcgtuEWSFt8rb2wL6paEvHtHPldGvE19A6xN8qbVZDlOSbycq9GZr8uNDeOfU4zN1NEgyoEdFDNfvP4SLtRQXWy3xJCEBeSdD5TG2X64lGkEU+0MbMf2afEXYO9tKV7E8Xzkp4P8SCwlapOXf6NM7aStWZiD4r4JtChDIdQ+pId3Rd5+eYFSFgpIRLglWhwUOfpHyM1VM+hbhEdEQTidqpA6SBp89VaiLwd2ELoXogLhZgsaZvmQP9xqiETZmYVKfifF3uI9rtS1pMKvRDwKMR28JhqWzRzGzSawDnBVxHxbsrxfq6nrxnovMg/3pwu7LlWToiEaleb/brQ7B38tshtnUQupCgCracKkWY3KqqJNrEjYFohimCynGqfpk8n2RsTS11AM1sGpOBf6YS2nxTwqFLsCekxxO2EXP/YYpvpwmZA8GmIt4P8TpP654x1XOCVaCxFoVZWiAT4P2lyoHG+jPiiKOuQcjviTqoWpXa8pILdmKD6cG4U3A9qE2moSLXRsjHZoE+n2RsLuteoma0wUvC7s120dWwvonhUSc8hZgIk7bgXzBaXqLZ/vlC1QPJWUrzezMUnjHWcXaoACSsxRFbj1EbpO0dZRiqUOgW3hjSliNuBbVQvuD/UNl+tMbW9EHcqeATKkRpqNKcnSvp0zkHSbA37A23QcUtRpocFzyPtgdg5EyA9Jt8WU6sre1DwZRCHQvndZr34jIHOJQ2QsBJDZMsnW8e54/yZsrO9KCJqIl0U2hPVDgGdzH6w/eG2+Qiqz0NvKD8iRVtG7bUi5+b0BA6SZmvUb9TB+aFbi5R2V13Y8ezMWpBdOEDa4mmthd0ERgO+JuKdyPnNkvwJY53nljpAwkoOka9GZp/G2N1/omy0TRcpTaPopGqJvJVqv9KV+/vZUqlD3BpBh6QyE0MUzWkmR6fp04WbuXSCmS2x36iDtsltRVF/QDk/I7EHtJNq5zSHR1tsTWAw0HFFHImsd8pUfsDkhjP8KiaXunCw0kPWq5GRxvjd2bNl0f1VgTZKrAd2Q9xNteirP+h2I1otkpsi4j6kwUQ0cz1NwMAUMLTE5TOzm6FPnZQTW4qs3cr5WcGz1VqQ3g/bbgoBYxEcVY4jQbxdRv6U9RtO859jaqkL17KyQyS0xkiO0Td0qsy1I0XWlNAgQRtVa2S9dU/8obf5aXVR3QZ6QUS9oBwrp4pBDmrsZu0EYGZLZL/qlOMba1n356TnAv4O8RDQQ7U0mNliae3C1yC4oMyRCA6WKR8hrfueny+fAAmrIUS2bO0d5vTI8ZKUi6R2iU0QBVXXdg+r6Xe1m6Ud0pZA90scL+q102X/xAR96vf4SLNVar/q9I5vrSk/kEPPSDwz07O1jtntDM0WWmsCTQaGgX7BJwkdKlN8xPT4Sf5p3bLowp5r9SyF83I0uXXdIHWdLOFziHdAc7dHNLsRMy3XagduFfG4lF8qyuZTMLGd/apf7wnMbIXpU0Hn6EaUd2X0rMRPQ/EQsBFPorHFl4ExxDHQ2yH+WBbF+4xNnuCX28eW4y5qq6t17uVo0qeLMH4s5zJSEQ2yAmITQSezV5E+Edh8zN1je3coeiTW13Keam4cnUA6vxw/1Gb2FzioGsdH1tOmu5OKp0S8GKFngF6qutL1hi2W1laGY8ApIj4I6d9LxXtMjB/jv24Z5b8tcQmvYnWFSIC9UbJf/Wwcy7nMKUXqFtoR1T7J68ELw9oNabU+bBLqIqhn5e/JxUX+ZaiJNOggabbC9amYCZD3JqU9oGcjuB+xjmpcvesMWyyiaoEcBY4jPgO9W5KO0Db+Lf9l67JsgWxZPd3Zc70SDei+SFn7NgVHAv6X4HXgONWU+db6S2bz0Zqx3QXcAuwuVD5OUb+b/3lxHfu0Oj9HZmvGQA+14q6ktEfE/0bEM4jtOEDa4moFyAmqoXdvRuh/5ZTeguZ3fLJ1fDkHSFiNLZEte2Oa19TfHJ0QjbKRUowL1aJaAmgzq/l3t8USQE/AA1JqFJRlSdskO89+R9UNYWYrSWs/7NHRnxRRPirFc4Eeo5qQ6Uk0tthKqhbIc8DHoXijVLwPk9/x1aZRXo18nccvudUdpH4eU+zTae4dzwm1ZdgUweaZWL+RqovbJwmbj9Zx0qWIu0A1KSaLxFBZtI/zmprLbekFM7uOP9DGyND2ImqPCp4neBxiM6hVN7p+sMXQaoGcBE4KPg14uyz4gOnRo/xy+4pplFjdIRKqBclf08XmheFjRRE9UqDQcBCPUe1u46tNm6+gOl56gZ8EPCwYKEiT5dD4NAd1zmtImq0Qr6mdoYlbipQek/Q3wAvADlAHrhNsccxdxmeaiH5yPpwSfyopDtGc/JZvtq2o1WRWf4gE+I9M8+vps2V3R73IFJCKQL2CDqrJNt5n2+YrUY2T6hW6U6QRcpqqkcebp8dK+nTeWyOaLXP7VefC+KZayvfmiKcCHhXsoBr37MXEbTFVXdjBeYkvIsWhknSIYuoon6+MLuy51kaIjBDSKL8dPF3W67Ui0SboCigF9wBbqWZtm81HAHVIWyJ0v4JmVowQ5TiNoWn2aWilnQjM1ow+FUyNb6nV9FCGlyReBHZS7XDmSXK2mDIwDhwn83EEh7P0Lmoc4z9vWpFb6q6NEAlVkOzTCGOD35dt9aKIQMGkpEZULZI13LVt8zNzjKiTzG0RZOBikWOwjGKC3f0NqsHSZracHFSN/uFeanF3VlTbGcIjVD1S7o2yxSKgCTFN6ByZjyPpjyXpEJPltxzbMLzUBfxLrZ0QCdUakgc1yumRE6UShUhAJ7CJoMbs9og+kdj1xMz/24Ftge5XxFAhTZaN9gl+o+/5VSy7LarM1qyDqnF6ZANRuyuRn1bwDOI+qvO+WyBtsbSWFBwDnZL4MiLeLUnvU05+Q/emsZXcc7W2QiRUu9rs1xAdIyrbilpB7pJSDyiAu/D+qHZjEtAluDsAQapFMdjsHh1Emlrua3yZrQn7lDg22ENH2pmkFxX8LBSPgTbiMZC2eFoBsklwnqz3IuLdHOU7FBzl/9g0stLriLV59fVKNLhj3RCFTpQ5vkB6H/QRcJJqyn3GC5Lb9bW6vwpgk2CXgkdz6D4acTsHBnq9ELnZEpOCXRd7qNd3JKXHBX8b4imQFxO3xdLKD62tDPuV9RXEOzmlt8j6kk96V8VuZ2uvJbLl5WjymgYYnTiWmmUtk9qFugI6IbbPLPPgAGDzEVRrjm6MiDuRHikij5Rle7B74CiwIgdMm614+5Q4MLCOaL+7iPyE4IUQjxHcyuz53SHSFkNrJ5rjivg2xDs56TBT5VGOrl81ky/XboiEaumfA539zeZQUavV2rLUrhT1kEDcStC51EW0FaFVCSWkjYgnFbQVlLUy1xv0aZpfMLkarjrNVpSn6GC07c6U9GJWvBzVJJotzC7rZrbQMrMLiZ8BvR+Zt3PoPQp9QWfvqgmQsNZDZLX0zyQH1p9p5qFU5FQn06ZQPSK6qCbdCJ9sbH4C6CbYCdEtMVXAQNkYGue3609RnVTM7GbYrzpjg7cUUX9YmZ9G8CKwnvBOZbboJoB+SV9HSodzxFtMTn/Nf94wtNoaE9Z2iITZIPm7yTNldNWKRB1SW8BmoS5mF59117Zdz8z6kfQAt0bwkBSDBUWUnSM1fq8T/KcYX+Iymq1+v1cX40O3FmXtSSW9RLCbqgXSdZ4tBs35swGcEXwccCiTDyGOcWzD8GoLkOAPVKV6Y8foO3eyzJ2pKNWmYAvVOLcdVPtsm81XAB0SdwfKgq6ijLIcmZyiT6fZG9NLXUCzVes1tXN+aFtRKx5X8PeIPcDtuPXRFle1lSEMi/gygjdy1ps04ismelZVF/ZcDpFz/WLrGL8+f6Ls7mwrlNdJQRDTQrupurY9jsbmozVje4ugPUKdirhYoxxqluMN77FttkheUzujE9uKejykrOeIeB64m9nzt9kiiUmgP9B3SIeyeIc2fcbWdcOr+XzvEDlXa3vEX58/WnZ3UmSmlCRgM5nbCDpa91zKYtqy1wqRCVgn4jaJRzJ5pBZRNvtHG+zXIK9EY4nLabZ6HFSN/oltRdl8Sjl+GhHPzeyH3Y13o7HF0VrKJ4MuCt4l4p1Mfgv4nE96B1drC2SLQ+TlWkHywMCxUu0pST2CrRE0gduY3V/VJySbjwLoDdglosxSQck060a+ZJ8urvYTjNlN0ac2zoxvKWg+IqW/qQKkduLdaGxxtMY2ZsQUwcjMTjRvZ8o/Q3zDresG+cfVf353iLySKkgO8z9Gjuc63YnUCZqiCo53Au1LXEJbORLV3uy3BbSJUIEulCmNseviNLBi90w1WxYOqsb3oxsouFcpPR3wlNAuqv2wHSBtsVTrQAanhb6JiHcyOsR0fM3RdQNrIUCCQ+TVRYg+DcLoF3m6bKZIU0p0hlgPbGJ2j223SNrVzK4fCb0QbYHGpThdZI2UtfoEv9G099g2+wtJwW8He2hLd6UonhQ8DboH6MU9RrY4ZrqvaSAuKvgwSH/OOb9L4gs61636Luy5fJV2LXujZGvPAGV8lYOPQnwQ8CVwgWoWltl8pOpL7cD2gEcV8WQhdtM2dBt9alvqApqtOFJwoL+bzrZ7UqSnhV4K5UeAzVTDSBwgbSHp0peYAvoVfB3SuzmXb9Gmz6itG2BvlEtczpvKLZHX83I0OaghTk0ei2gcElEIjQbxJLAVb51l89OabLNesEtSE2K6KOqTJRNT7Nc5T7Qxmycp+N3ZLmpdPylUPqqIFyLiScQ28GLitmhaXdjHBZ8FvJ8Tb1Ckb6BnaK0FSHCInJ+Xo8n/OHO+jM7PipoyOU0F2qqIHqrxkX4dbb7aqSZqlYqYIJfD5BinazyzX+cdJM3m4QAdtHVsLxQPK9iDeBTpNhwgbfHkmRbIi0p8Hpk/5qTDiC/Z2jPIy2svQILDz/z9cvsYv9epcmiYolBdke5B6gZuBdbh7hO7vtbxUQO2hXQfxEgipnNRlmwcbSKdX427GpgtmP2qU05sKUi7lfUCwR6Iu6h2i/I52BbabBd2qD+IY2SO5JTfpSg/5xcbV+VONPPlMZE34h+YIJrnSuKrEP8f8P8GfASMUzVzr9kDyW5IUAXJW4EnhV5KEU/SyHfzL0Mb2Cd/Ls2uRArWjW2q5Xy/lJ4n4u8hdlEtJm62kOasAck0EedEHJL0h5T4E430LVs3jq3lAAkOkTcmQny1aZQ8/X2Z+SCIdxR8CBwDRoESB0mbnwC6EDsieAzxSJGK+ymK29l1scdB0uwyB1Xj/xnZTM735shPCD1BtZi4e4JsseTWTjRCXwd6JwevNxt8yo6ei6t5J5r5cnf2jXo1MtII/zJ0oox6rVDuENERxMTM4rabl7qItmIkgk7E9oCHJE0URFmq3mA3x4GJpS6g2bKwT4njI+tp475UpOclPR8R9yF14vBoiyMD46ATCr4McTgRhzLxFTu6HSBnOET+JSLEPg2xe/i7sqxRZE0r8nBAIWIdPxyb4xOcXcnc46JDsAvRISkVqRgsp8aHOaiGT1S25knBgYF15PY7U9Yewc8CHkfaRnWuNVtIrW7sBugixEeR+VMOvZdr8TV0X1yrk2iuxCHyL/VqZPo0xOTgt2V7vSxIIdgI6gW2A11UXSxm15OA9QQdAVOiPEHBMN+PlhyUr3htbTsw0Mt02x1FUT4u4vmAh6iWV2tt+GC2kES1DvSAiG8DHcnBeyh/zsD6UV5xgJzL467+Gnuj5NiGYWrT35WKz0BvE/GnCD6lGiPp8ZE2H0HVotIj2CHiqUR6lqRdnB7xRBtbm6SgT51VgMzPKcffA08hbsVL+djiaE2kuSD4OII3UkrvUUvf8uX6IS/B9mNuifxrvRqZfRrh/pHvcipSUZZTIhoEm6lmDPpkZ9fTOj4EbAh4VKiWIjWzYpLdAw2kNb2MhK1Bv6WdtslbisTDEi+R2IO4naB95h4+r9pCEtXk2PEIjgFvRU5vNevxOXReXEtbGd4Ih8iFUHVtD9AcoSwKUs7rROwIqR3YQtCBW33t+oLqwmNHRNRRHimC4bLRVnJg4Fv2acQnMlsT+tTJ5ND2ItLDIp6FeAzFrSBflNtCa42BLIFB0HEpDgd6t0zTn9Kz4Sw/D29zfBUOkQtlb5Ts1xBd4ydSEZ/n0HoFUyEeQtzhIGnzlIA2pE0RPChiKoXacm5P7B74xi2Stuq9pnaGBrcV9dpTgr9DPA/cBerAAdIW1twAOSBxJFIciqx3yjJ/TMeGs/xHHCCvwSFyIb0SDX6jgWZ96GihaCOnUFCLUCfEdrjUDWN2LQF0ibiHai3JdUVZTpW5Ns7vzjaRxh0kbVXarzpD45trUb8vZ/aAXgLupmqhTzhE2sJqjYEcCek7UrydIv7cTMVnbOw8x89jaqkLuNw5RC60f2aKXzfOlZ3t9SKpDaVOQVdUJ8AtVEEy8MnQrizmfG1AdBCEiDMFGizbOhr8lpPA5NIW02yB9amN6YlbaqGHcqkXFDwT8BOgZ6mLZqtSpmqBPB/wtSLeC+ndZlF8Dp3uwp4nh8iFFiH6NM7Y4Jmyvd5WQAdETYiZIHkLft1tfmJmEsEtEg8TMVooGmV9YoqDOuOlf2zV6FMBo+trKR7Mmb+N4EXgPqB7qYtmq1KrBXJM8AXweqC3y2b+mKnOc/zKAXK+HGYWw94o6dMIzZFTpYp6geqgWqTokeiiurIu8BhJu7pWS3UCuiO4TwKUmkQ5wrHBUfZp2BNtbMU7qBqnhtZR1O/KKp+JiJcEjwEdeK1dW3gCmsAYcDzQ4Qi9WVL/kLKzn1+Fe3lugPPjSZ0AACAASURBVEPkYtkbJQc1SP9olI1cFBGFRFdIhSLuAjbgEGnzUwe2RBCCkQTncmd9iDvOHwVGlrpwZn+xfUqcHttMFA+ksnxOkV4A3U3VAunzoy0GAYMBnwsOB/FGmfNn1DvPugXyxvlDuphejiYDPYPUi+MlfI74QCl9CDoOTDE7M8zsaoLqc9oObA7YiXikyDxCd8ed9J3rQfL4Wlt5+lRwz8hGIt+X4CVF/IeQHkFsYnYSjY9tWwitujYDoyEdE/FWiH8vkz6inDzLXgfIv4RbIhfbK9GgTxeJ8ZTJbUmqQXSAtkDUqRYjN7uWVkXaDtwaiUckNYqSZtnoaPCvfA+ML2H5zG5MnwomRzfTxr2J9LTg2UCPEmzAXdi2OJpUO8l9pxTvh3SoJH/E9MQp/mnbOL9c6uKtTA6RN8PemOagztE/EblZkiLaldkaQZ1qD1iPj7TraQXJDRIPBFFX5EZRMFmODU7Rp1O+krYVYZ8SjaFeasWDSfHSTIB8COhldj9st0DaQpltgYTPBO9G5q0y6wPae0/wX9ZPOED+5Rwib5aXo0mf+omJlJQ7c2KjRAQ8SDVju45PnHZtQXXB0St0rxQjwEihtumSQfi9zvGfwi2StnxJwYGBdZTtO1Mun1PE3we6H7GB6qLaF9O2kC7Nwga+E7wb4vUy60M0foK9vRNLXL4VzyHyZtob0/xG/c3a8NdFER0QklAEiapFsvV+OEza1QTVcdIbsEtompRTQYpyZCjxG53y7EJblqTgX4Y2ULTfm8h7lGJPiPuAWwh3YduCawXIi0F8K+lwwFtl6CM0fsJd2AvDIfJm+2em+N3E6bLspEiISClQh6qWyPVUf5pdSzXZJtgaxCPKtJFSWYsom90TTfp0xl3btqxIwb/SSarvTFkvCn4WsBtiA8g70dhCawXI8YCvRX4zUrxTSh8wMXWS/7Z9zAFyYThE3mwRQhrnt5wo24ajCHUI1oWiXWgnsBFv72XXF1STsrZHUEgxnoMxGuU45egk+3Tea0jasvEH2hgc2lbU0xOZeDngOWA9qDUG0mwhZYhJQmeV+SASfy5z+QEb15/kH3u9leECcohcClWQnOJAea6cLj4tCgqRpwIKER14n1i7vtb4SEH0BroPMZES0zmlce44P4E06j22bUlJwe/OdjE8cntRL54QPB2wk2qd3BoeA2kLq7WUz1DA1xIfBvFmWfIFefIsP9/gALnAHCKXSrU94iiNwe9L6hQRWegWgvXAdqALL3Vh86J24FagRDFaoP6yo36RAzQBDxy3pXOADorO24ocjyv4G6SnIG5ndha22UKpurDFFMFJSYcjxRslfICmT/DNNp8LF4GvApfS3ijp3jBCmU+VRXxGxLvAe8AxqnX/SrwYuV1bq0WyG7g90IMKHisi7WZyaDu/UccSl8/Wqv2q05i8pUjxoNCziOcg7qVaysfL+NhCElV9OUzwLdKRiHirLDlCbfo7vto06uE9i8MtkUut2h5xiFNDORNvFxETIo1D7oVow4uR2/W1drXpFtyprKeJFEVduayPTwGnlrh8ttYcVI3+0Q21Mu7NiqcCnhLcA/TgAGkLLwNjIX2tFIdC6a0y6TC1iWPsvWV0qQu3mjlELgcvR5ODGuH70WNlPUfKdCvirqjGRm7ES//Y9bVaJDdF8HBAj3KaQOUg//eFUV+J202zX3X6xzbXSh7Iys9IPEPE/VSt5R6iYwupGgNZdWGfrXaiiT+Vkd9nbOokx7d63dxF5hC5XLwcTfZrgPp4LYV6M7FN5OkgPQC6nerk6xBp19MObBIKBbsTnMpF+yi7B46xTyMOkraoZlogKbUrZ55VSi+GtBvYjAOkLTwhJgn1K9KXIX1QZj4mj3/Pf9027omFi88hcjmp9tkeaE4OfV3Uow2lMSkrInqpxhF5DKtdS+siI0F0B9pJpOFUZnJuE7v7j1LtHWu28PYpcXpkA3BvivS0ghcj61GCbThA2sLLQJPggogvA72fgk9zyqf5Lw6QN4tD5HKzNyb4jU6X9UEVqolQT4Q2StxNtSyGtwaz60kzM7Z3gNqI6CrEWNloG+U1Nfh5eJkLW1h9KhgbXAfFrkR6VtILAU8RbGJ2S1f3pNhCaC0k3qDaD/soxKEo8ztNal8z1jnkAHnzOEQuR7+KSfp0tmwO14oi2kQ0FDwamScItuN9tu3aWhNteqm6t0slHa1Frb85OjHFQZ3l5WgubRFt1ehTAcPr6Wi7J5GfkfRSwOPAJqqJgQ6QtlDmjIGMc5K+IeKdyPmNsig/ZnLdeV6JxlIXci1xi9Zy9Qsm0cTpMuWPUo7XQ3pbwTfAyFIXzVaE1h7bXcA2we6c82M0y3s5PbaZg/IFpP31pKAx1EujdndSfgrxfASPQcy92HWAtIVUEgwJfRWJtyPnN8q22odMbjjDr2JyqQu31rgiWa6q5vgxfqOTzbbJZpFznaTNQRSCXXCpm8gXAnYlcyrv6AnpfoipRDRy5AanR0r6NMDeKJe0lLZy7VPiwMA6UvtdSflp0EsQj6G4FdSBd92yhZWBaardaI5BvB/kt8uofwQdZ/hVTC9x+dYkh8jlruraPlc2pr8oFIUSI1KMBHqGKkg6RNp1qANiByiBcsoxlYs0CQMNpGGPH7IbVm1n2Ml0zx1FlE8KXiRiD3CbA6QtkgwMRHBU4lCQ3ywb+SM6Os6x1wFyqThErgS/YJIDvSfKyaFmQb1BqD3QNlWLka/DJ2y7tjQTJG8HNYiYKEqNldSnOdB/DM/Ytht1gA7aum4vMg+LeBZ4EriNam1bn49soWjO12QQ30s6HPBGqfwhHevPsDe8neEScohcCSKENMmB9WfKPFIvclovRW8Ek4J7gVvwSduubSZIcjvoCQXThaJZTrVP0qfjvpK3eduvOpND24t68YSknxE8DfyE2QBptpCmgcFAJwRvB/y5TDpCWn+aX+AxkEvMIXKlqLocJ+i7eLak7dNCUZPy1MwYyS6q3SDcAmBXMmf9SHqAu0GlpNFaUVxsTo2P0acLDpJ2XftVp3d8ay3X7svKewiegbib6rjy+ccWSmuITQaGA74W8f5sgFz3vVsglweHyJVm68Yx+oe/Lxu1WhEpKViP1EPVErAOv6d2bQXQA3GX4NFMvlCrxWRzavwLDqrfS//YVfWpjXJiWy3r/pz0jBRPBtyBA6QtrNY6kNV+2Oi4Io5E5o0y9CHT4yf5p3VugVwm3PWw0rwcTbb2DtMsj5VRfg56H3QYdBQYZ3b8iNnlWjO2E9AT8ICIv8niZ7Vavp/+4V72yecE+7GDqjE1vrmmvDuH/laZvw14iGoDBG/JagutBIZB3yji3ci8XhY6TG3yBL/cPubJgMuHW61Womqf7WHS+PGccntKUZCVCHqplv1pn7mnT+x2NTVgS8DjgiITI+RigF0Xm0gjPknbJX0qOD22mVq+Nyv2CF6M4AFgI9Vx5POMLYS5k2jGAn2riMOheKOMfKRqgdw2vsRltMs4RK5Ur0SD13SR0Yljqcz1nKJNUk9UH8DtVGMkvV+tXUmrNbIDaA+4H+UzRU4XytTW5ED/d0i+2rdqLUiG10PMbGeYn4uIB6mWF3P9YQutWh8ZTog4EtJbZeII0+Pf88vtY/xyqYtnl3PX1Ur285iio/N8Mze/TeLjkN5GvA+cgJikGlNidiUx52s9xMMK/rYgnmaq43b+QNsSl8+WmhTsPNtJLnakSM9I/CwiHgE2M9sC6VZIWyiaqbfOID6K0Jul0vukxvdugVy+fCW50v0DE/zr+jPN8aGoUcs55yyiLdB6ZvetBZ/s7eq6IXZB3qhIuagxUJ4fHWCfzvNq+EJkrfot7XS0by2kB7J4fmY/7FvwGEhbWK0u7AbQr4gvAw6XEe9Rnz7GLzZ6Q4RlzCFypavWkJzg141zzbaIIqKNoFvQHdVpfjOzYyTNLhdAAepEsTVCD0hxslZjtLlz9GP2a4BXorHUhbSb7PfqYnz0nqKsP6akFwMehNgEau2HbbYQBDSpurBPIR0JxTs55cNMlN/RvXHUAXJ5c4hcDSLEPo1xx/mzZXdHvRAdEB2S6tXJn614fKRdWxC0S9wptCdHNKirAaNfzKwh6T2214o+tTE5saUo40lF/nsUTwE/Afli1BbDBHACdISIP+cUh0lxjH/eMOQAufw5RK4WVbfjCH0XT5XNtnoh2gnVg9Ql1InXcrNra41v2xLEw5JIpPFclFOMDU6zT8Pu2l4DXlM7Fye21mr5gYz2QOwB7sFd2LbwMlUX9hnEh0S8mSO9R9n4hs/WD7HXAXIlcIhcbX6xcZjfDh4vO9tqRU6hUA+ik6oi8L62di1BNY72liB2I10sIobLen2Cp/iGah1SW60OqsaZiVuKmh7OpV6IxOMSt1AtG+Zzhi2U1hjIJnBe8Gkk3sikd8iNb/jH9YNugVw5HCJXmwjRpxEYPlaWtUihnqobKkQVJLtb91zCUtry1DomasBGgsdEmiwiN8vx0Sa/13f8AxM+wa9CM2tB1sRDOfhZBD+T2EnVg2G2UOZOohkQfBXwfi51hLbmUYr1nkSzwjhErkZ7o6RPQzRHjuU22lNEgQQR3YjbqdYHNLuSVpDsAHaAppViqsiMlCOT4/yh4wwwtYTls4W2X3UY21yTHsyJZyT2AA9QbaPqsdS20KapWiA/j+CdpHgnp/gauoc89nrl8TqRq9XeKDm6bgClb3KpI0QcljgGDFNtKeWrPbuaS1sjAneHeFzi8Voq72V0bCMH5YvP1WKfEp2jG2sNHs6JnynrhUA7mQ2QXgvSFoqo6p4hwZch/j0R/9Ys4wNq3ecdIFcmh8jV7NXIjHRfRMXRpPgsxBeIk8wGyYzDpF1Za+xst2CHUjyexWOUeRf9Y1u8x/YqsE+J3cMbSNqVQz8N8XcR8RTEdqpeKo+ftoXSmkQzFMG3IR1OwevNQh/S3nXOAXLlckWw2r0SDdq7BptqfhPwZ9C/qdrV5iLVwGazqwmghtgU0kNCzyelp2jmu9k9sM5BcgVrBchc25kinlC1h/o9wAZ+uEmB2V+rNYlmUPCRFH+MxJ+akT5na8+AA+TK5kpgLfiEKerT58rQkYA/R8R7oO+p1udqDXQ2u7KgA7E9gsdBTxfBg+T6bQ6SK9Q+JXYPrCPXdiblPQQvBOxG3g/bFpyoWiEnESeCeDdS/HtZ1I4w3NXPy+GGjBXOJ4y1oFrfb5T/q/9k2dlOAesE26kmT/wE6GJ2L1yzuaoxcUEn4g5gRDkuFInhMtdhd//3wOgSl9HmSwp+fb6b6c47UpRPQLxA8BRwB3GpPvB5wBZCpmqBHAVOEHEkQu+UzfQRt3eeYa8D5GrgELmWHN8yxj0j/WV7fFGINomhIPYI3UvVjdVqVXIlYpeb2R6R2yJ4WqJWZGplo73Bb3SCX8XkUhfQrkMKDvR30965o0g8pkh7kB6DuB3kxcRtoQkYCfhS0luReLts5iNs7nEL5CriELmWvBqZgxri+MjRsp6mCmlcSe2ILqr9tTuogoJwhWI/FsAGVVtp9io0UUvFQLNjfJz96vce28vcv9JJs+MnRRGPC/2UaimfO6l2tPKwBFsIraFRmWopn37Bh5H4Y0nzQ5obzvDz8BJhq4hD5FrzcjTp0yBjg2XZWY+izN2KiCAmVC3tsR6vDWdX1trRpg0oRDyakwZr0Gz2juMguUxJwe/OdjE08pOiFo9JehF4DriTaiiLLxhtITWourC/I+KDgNdLmh9SbDjFr2J6qQtnC8shci2qFiMfoTlysqSoFWhKaIyqJbKd2V1tzOaaGzY6At0nRc6hoobKZsfIFPs04D22l5kD/d1E1+1FiieUeYnQs5B2zrRAeh1IW0gCJoHjBK9H1utl1vts3nCanztArkYOkWvV3ig5qEH6h3PZKMoC1RRxG9CD2E7QjteJs6urQ2wP6JZIOWKYtjTE7oEm+zTiILlM/I8z3Ux13F4U6UmFfop4DmInqAMHSFs4rWV8poBTSB9FxJ/LojxEff0pd2GvXh4Hs5a9HE0+6R2kmb8vkz6DeBPxFsExqvEsXv7HriaAOrA+grvJ+YmC8gmabXey66L3W14OXlM7dNxSpNgt6bkqQHIPVU+DLxBtobSW8RlBfAn8OeDfytD7TKw/yd6YWOLy2SJyS+Ra92pk9muY+tixLLWnoBTUo9r27BaqoODKxq6kuggVW4l4TIpmQVaZ2kr6dJRfMEmEL0KWwmtqZ2jw1qJWPKyIPSE9LbiLaitLB0hbCLr0JaYITpJ4OzL/Xmbepxw/ya96vWrDKucQadWuNgd1gf6Jb1KjLDJsULAlquNjE9U4SbMrCapgchcIRUwWYrhkYoQ/dJ6l6t6ym2m/6lwY31Qr6g9k9KykPcAuqgtD9z7ZQmm1QI4RnAU+jBxvlbk8RLP3BP+8bopfLnEJbdH5hGKVl6PJWOeFZlEeD/gk0OGAL4Ehqn223aJkV5OoZvn+JMRDEo/UyryLofHN7Fd9qQu3pvSpYOPwOqK8J4eelngu4AGqVRdaLZBuhbSFMgmcAL0Z6I+leJ/1vaf4VbgXYo1wS6TNOsY0u8r+slZ8WSh1SCqoKp8uYO5acq6ErKV1LCSgW7BTobEsTaJo0DEyhXTRFcpNMn2xm2i/IyU9pqwXImI3XNrO0J9bWwhz14IcEXyZlP6tjPwOefR7/lPP+FIWzm4uh0ibVS1GPs7pkTNlpI5CuU0RPSE1FHEHsBFXRnZliWr87OYgHhBMpWAs14oxfjtYsk/DnrG9iKTgf15cR9GxM5GfJeu5iLiP2QDpXidbCKLqmRoHTgo+DcWfytB71CZP8I/bx5a4fHaTOUTaD70cTfZriHVj35dJRSFqStEI0RTcD/TiLjG7sgDaEdsjmCbiYiENlZ21Ke44/x3SqFskF0G1H3YPnR07CsonpHiJiMeBbVTB3mwBxRToFMSbifxmWaTDpPHv+GSrWyDXIIdI+7FXosF+XaRrPJXRjCKBIgpED9Ux00W1q42DpM0VM//vAG4N8bBCI0Vmsuxun+YA3wNe7mOh/e5sF+3dOwriUVVbGT4G3EE1BMWfUVsIM5NoYhJ0VvBxUv5TWcTbpLHv+cXWMfb6AnEtcoi0K3slGvTpAlPjlFGmlKkT6oAogB1UQdJLhdjlWq3U3UL3BpRCZVEyVaaJSfbrtLdGXEB96qQxfFuR4illvQA8Q3AXPxzDbPbXagXIb0FHgvhzGfEu02PH+adt4+5hWLscIu3q9sY0fboAEymrrCelNkKtFsi7mF36x0HSLlcDegW7REwBo7VGOdxcNzbFfl10kFwAv1EHjclbiogHhJ4neBYuBchiaQtnq4Dm/NkgOId0OMTBsuAQY5PH+a8OkGudQ6Rd296Y5jWdZ2giJcoiEyHUHqgLYhvQNnNPB0m7XA3YFLBLwUiOGEK5wbqxr+jTefZGudQFXLH61AYTt9Sa+cEc8UwonhS6h2otSH8WbaFUk2iC/5+9O2uSq0rTPf9/13aPORSSkBCTQCRTgpgRc2ZWUkN3F3bq3KX6dNmp6qJv8qa/g5Kb7ov6BFx0UZ10WZsFZsfaMrup7nOqjiozGSQEYkgEQoAQSKAhNMQ8ue/19MUOV4SExCCF5DE8P7MwSeE73JeH3Pd+fA3vOiHpw1DsKoM99PR9zn/fN+UAaQ6R9t2ejRle0YnmmelUo6kc6hb0RRUg1zMfJM0WCqoh1WsC7pXUSCTlFDOMnZz0QptLtEOJ5tiamtKdGT0teBp0G/PbGZotlingK8S7EezKkfYwOfkl/4MDpFUcIu37+Utm+eeuY83xqWaR1UXOvRL1CO4FNuD5kXZxHcANAR1CFMqjZW/nCL/lCJIvRj/EDiVuPtmLum/KSU8EPIO4m6oH0kPYdrkW1oAsgSHE+5H0L2WZ9sD0F/xPG/zhz85yiLTvpzppzDCoobIc/bQgdRNRBOpWVUZk4Y4YZi1BFW4KYEPA3VKcKRSz5fh4nZcmDgGuLfd97FDirrH1qONHCT0i8TBwK1XZLZ/LbbG06kAeAd4HXisjvc3M1CEHSDufTzz2w2yPWQZPHC1zd60oVShYB2dL//Ti3hC7uDpwA/CEyPVCKZVF1xSv6CueDe+x/W12KLFleA053Z5S/FTwVAT3ohgAeQjbFlMTGEJ6G3g1p2IvaeYQX26YcIC08/nkYz/c9mvHmZg+UhZ8RPVJ9UPgq6oEhPfZtotKVD3Wd0E8JfRYrajdzvDEWgblDx8XIwV3nO6jXt+civSgSH8S8BTiVpBrQdpiaO1EMwOckfgM4s1cpN1MTn7ML9Z5xym7IPdE2qXp3zAJo19nxbsph0gxjVRA3AjqoTop+eJmC7VeDwWwVsFdWdxP5HGaYyXSKfd0nEcKXjreQ9F/R5HKh0V6PNDtiD7C00dsUcwVEmcW+FrEhxH5jRzaQ6kv3QNp38Yh0i7N9ijZqVGGxj/JpNlETiI2BloDdDF/cfNFzs4XQH+gOxCnEzGZlWZ4eWiaHZp0j8cC/0gn3T03ppyfysTPQ7oXuJ5wRQRbFGK+F3JEsC/Qv+SsPXTwCfsHRvx+tG/jEGmX7ploslNnGBonN+hJoWsEtahWi24g6G53E23J6kCxkWArxERBOVnm7knuOP0VOzTuCxfw62O9dIzdWCjul/RoEA8AN1Ktdvf+9Xa5qh5IMUNwWnAgpN0Z7Sb0CfscIO27OUTa5XkmmgxqBE0dTOSOHEwoNBOwjepi59eYna+1x3YncBPSjIiZIkezrHUU3HzyC2CsrS1stxdUpza2uSjTI0JPQmwFrqFanOS57LYYRDWEfVLEHyO0JxG7coPP6F7jOZD2vfgCb5dve8zyok42uyY/JOfZlBgIxSZVcyNbK7bdc2ILtQqR9xDcChRS7i1yKsqOzgav6NCqXbH9orronbq2aKRHFPkvIB4DrsfFxG1xtIawm8CwggOh/IeseCNHfMzNfSM8492k7PvxCckWx3MxTdFzklR8HtJekV8V7AVOAN4n2S6moCqUvYWIbZIeLmq6nZHJa9ixCkvXvKguipHri4buFfkRFPdB3Ez1O/KHflsM1V7YcFKwP4I3U0pvEfEx1/ee4plotruBtnysvpO0XTnbY5bG6Omy5KOAV+e+PqUamszMfwL2Sj+D+d7IgqqX7QYF90hxP5S3cdfYenZq9QSnV9RJMXJ9UUsPKOlnEfEIwU2gLtybb5dn4bm3BM4IPgzxWpBfb5IOMNZ72gHSfijXZrPF9Z/+vsnHo9Mqi6mUaIjoDqmfiNawdqssiS+GtlACUkTUBErErKRJzsxO8sD/MsPvnl/ZHzxeUJ1y6toipXsV8dNAPxHcSbU3vUv52GJo9UCORPAp4tVEvFpG7QOK7uP8j+ERI/vBVs+nfLs6IoQ0wctDR8tmV1FE9AmVgomArcBa/Lqzc7UCUo2sTRHcB8wmUjN3piZbzzSRRldsrbpBdVBObqhJd2e0TeJhgjuo3iutHkizy9FaRHMq4FNJb0diT7MoPuJM9zF+6QBpl8YXc1t8rSD50vHDZfSoqDEO0UBaB3TjC6NdWBB0QWyeG3nLRdZY2ahN8FsaVPv5riw7VePI5DW1Qlsz+onE0xH+sGWLqjWMPSL4iODVyMWeMuKP9HWfYLsDpF06n6Tsyqh6jSYYPPFV2ejMBepRik1IAFuoFgp4WNsWmns9qAe4iYimpDNFLabLkdGSF/TFiuox2akaQ+PrKHRbJh6TeCKCe4AN+Nxsl6/Vc9+kGsL+nKx3k9KuZooPKbqP82zMtrOBtvz5RGVX1r6Nk2w9c6Isax8XlP0imlSvux8BnThE2oV1ATdF8KikRlFjqlw3PsJOnVkRk/8HVXB0bC1Zt6UoHlHiiUB3U9WC9IJHWywlMBrEQaF3ItLeZrN5gNmBIZ5zgLTL55OVXVnPR4Z14xTNL8oo3o3Qm4gPQCeAGeZXbZu1BNWUhz7B3UE8LtJD5LSFL8cGlv2K7R1KcKaPKLakSNsIPRXSg8Amqg9YPi/b5cpUAXIC+Eroncixq0z6I+XAUf6O1VmD1RadT1Z25W2Pko1rRkn5UFnyIaG9ingX+JoqSLrsj50vqAJVn+BHEo+mnJ+gg7s4OrYWaXn2YEvBzSd7yfXNBeVDBE8DjwAbqXZ48jnZLtfc+TRmgOPAh4HeKoN3ac5+yXMxvWIXqdlVt7w/0dvy0doesZj6MuXcl1FdASHVIDZRXUDBw9t2rgSsC/SARD2JnpxU8k+nZ5flHtu/pZvuzhuLknsV8SjwAOJ6gnq7m2bLXisYlohZQl8L9gXxVhnpXXLjMJ+sH29rC23F8adeu3p+QYOi+2ST5ieh9E7AOyIOACeZ75E0WyiATogbI+IRpXgy5XiAWsfN3HG6b1ntavOiupgYvrZQ3CP0SBD3AzfN7SHuD0+2GDIwTWhIwYch7c7SHmbLz6kPeD9sW3TuibSrpyr9M81vB46VI6OpIOpE6pSUIrgLuBao4wuqzWut4E9Af4gfE+lUIaZKOjJb+RyYam8Tv4edqjE0sbbWrN+RE4+FeFxoC1XJKxcTt8vVmhI0CzolxceBdufE68zqQ27u937YdkU4RNrVVQXJKf5h9njZ3VkrUI2EJCKqFbkDzL8ufWG1hTqAG0E/legpUqSyHJ7kBX21pEv/7FSNw+PrSbotJx4JxRNzu9GswzVT7fItCJCcFLE/gtdTxBu5wcfc3DeyIioa2JK0fIaCbOWIEF9umKDWPFrC/hB7Q9obwcfAaaqTIXh4286VqHruNgP3CT1YU/0O+ieW7h7bO1Xj6MQ11Lg7FelJEY8L7qDaztDzIO1ytQJkAxiS+Chgd0qxq0k6QFffKQdIu5KW5onXVr7nIzOocWaGvyrr9VSk6JBUC9CCPYO9t7st1BrW7gA2CrZm8nGCaQ6Pf8xOnV5SF8wXVOfw+DoKATBz4QAAIABJREFU7krBTySeDLib6rVdwz2QdnkW9kCeEXwcid1J8UYzpX3QfZLtHsK2K8sh0tpne5Ts0Ch3jhwpo14roKasRKgbogfowRdaO1fr9bAu4G4RU0kxm+vlLIeGZ9mhpbF4YIcS/WMDZP0oRfGwpJ9F6AHmi4n7dW2XQ1SLaKrdaOAzIt6OiDeaTd6j1n3CAdKuBodIa6+qR3IUxo+UjVwvUtQlBoA+4BZcO8++qeqRFOsj2Io0lUgTuV5McvPJQ0jjba2DN6iC6fFrCO5KRfG4pCci4k7QAJ4DaZev1QPZBE4J9hPsCvLucjbvZ8PAaZ51gLSrwyHS2m97lOzUGQ6PR6mcUirWKrQ2pC6I65mfO+aLr7WkudI4mwjuA0aLxGjZ0TnLS8ePUO3UcfXtUILRAQq2JBWPSflPI+IupGuJs7VQzS5VK0CWwHDAp0TsSeIPTdU+ouw6zl/i7QztqnEPjy0Nz0STqb4zqPZFgg+rGpLsA1rbI5qdr7XQ5sYgHpDyo0WNeygGNjCoqx/YdiixZXgNZe1HKdJDQo9HxINUtSC7mS9XZHapWotozgQcEHoryLubOX1Aveuod6Oxq80h0paOX0aDyZ6TzWbz48hpTyh2zQXJ01SfvNs/182WinPqRwr9KCIeFbGtlpq3MjXWf1ULkbe2M6zXN6ecH0V6OuBequ0MW7VPHSDtcrSGsM8EfCxiV4jflzm/x/ruIbaHeyDtqvNwti0tv4wGL+po2TVVFjkJqTfQgEQX1TxJFyO3hVrhbE21x7bGc8TXdKYzbD3TRBq9Kj0zLw/10tt1fVFyfw7+Ym43mk14DqQtjtYQ9mgEn0l6M9AfynrtPege4tnwaI21hUOkLT3PxTQv6kTZMVIrUtErqR7E7Fx9vQ24vp59Uw1YH4p7kIYKNFHmDvFbPkWaumJBUgp+SzeT45sLcb+SnghRbWfo16ktnkzVA/mpst6MiDdKlfug+5h7IK2dHCJtaXoupvmNjpVjo+8VREMpGiElcbZHstXD414eg2pYu5PgOohtiiiLTC7Hpkf4x67jwPQVedSX6aIxcl1B7WGFfgraBuk6UOcVeTxbbfLc12SgL0S8FSleL3N+n/rAUQdIazeHSFu63maarWuOlHkkF2V0KLGGTA/BzcAaXLDZzhVUr4kfIaFgspbKY82uqcyLGuK5WNwg+aK6KIc3FSndLfQ04icobiUcIG3RZKpKA18q0nuR2V2m/D71NUfYHkt/z3hb8RwibemqikZPMXjieNns2V+IHoWaATMLto4zW6i1o811ETyQSw0VhYqyd2ofO/X1ou1o84o6GRm+rlBxn4jHAx4Q3EjQhRcs2uVrlfKZRHxGsCfEG2Wh95id/Iq/XeMAaUuCQ6Qtffs2TrJl+HDZXVMhJpSpE6wF1rW7abZk9UncRahLOQ/Ucqj55dgE0unLnh8pBS9PrCvK4l5F/LeEnhBxC9C7OE03mwuQcJhgb8B/Lcv8HrXpr/ibTZP8bbubZ1bxUKAtDzuUuON0H3TclAp+JvHnQTwCWke1PaKHtm2hVlCcDdgr6f/OqfjPFDOfwLrxS94SblAFU2NrKfhxivgLQv8Oxd1zdSDBr0G7PK1V2OPAYdB7If6tLPgDqf+Q50DaUuNhF1seno/MjesnSTqRMx9GxGug14DDXKlFE7actRZdFRI3KuKhRN5Go+NWJob7kX542JNibjeaH6eUnlDEwxA3ze1E40VedrlaQ9jTwGGIPSH+rUzFXpoNr8K2JcnD2bZ8PBNNdmqYoanPcrOZk6KhRF+ItVTz4GpUH4x8MbeWRLAxggcRzUTOuV4veel4gx+6NeJLx3uo99+SUvlTKf8sIu6lmlLhD+N2uQRkxAzBCWBfKL9eBq9RTBzhP2yc4D+2u4lm3+QQacvLM9HkFZ3k1GTOKack9YvIAXcD11PNS/Pr2loS0EXmBoL7iRgv0GjZ0TXJKzryvYs0v6JOTo5cW6C7c8Sjge7h3N1ozC5VFSBhguAI8AHweknxLrWeL9juRTS2dPlia8vPszHDKzrN6amDibLIMEZoAuJR4BaqGpLgi7u1VHMWb0I8rNBIoTRZjgyXvKKj3xkkd6rG0NSGWkftjpy5L+AuSBugDftz20rTCpAN4DjwHsHvc6S3KRsHXcbHljqHSFueno0ZBnWiOTNZUmtOJlIS9EemG9gwN0/NW84ZVK8BAT0EtwUxLsVskYpGOT4xzU6dvGjpn0EVDI2uoZFuy6FHJB6IiBtA3XgepF2euUU0MQMMKfRh5NidxS5mZw9xaO1ouxto9l0cIm352h6zvKCT9E/kJPXkUK8iFHAf4voFK2bNgupDRa/gTqFZch6D4iSHhqeRRi5Y+mfsZA/d3Tel0AMinojgPqopE54HaZdj7rUWM4SOCt4P4o0c+U1y4yCH1o/P1ck1W9IcIm15+2U0eEXDzdNTn9RSqUxMS0oRdALX4d5ImxdU4W9tRNyGOF1kfV1216b4h5MlMHbO0b9RDyNjNxTSPQoeD+J+0DX4vGmXp7UKuwE6LfFBwL9mxW7q+hzWTzhA2nLhk6Etf9UcyaHmyGSuKacs1kewTtANDOAgafMCqCNtBO5R6ERRMlN2dczwgqb5ZTQAGFQ3I6M3FLV0n7Ieg9gKug7OlvMxu1RVIfGIIeW8P4I3chRvUpv4mO3Xjre7cWY/hEOkrQxVkDzVPDX6ZZHS+xJrIlGXuINqn+16u5toS0KrN7IHuDEUjylUFsFYuW5ijN9olOHjQR65rqgVD0o8EykelriJ+ZXYDpF2KVo9kE3gqKT3I6VdWXkXKg/xi40/rOSU2RLgEGkrx7Mxw+CJY2Wz+8NCFMpAqA5xB1VwcA1Jg/kgOCD04wiylE5T5lEmx47T0VsUWbcLfgL8mRQ3gVp7Yvv1Y5eitQq7CUxI+jRSei3n+D3TMwf4csPEZW/HadYGDpG2slT7bH9VdtaLQnTMDWkXVKV/1jC/IMJhwBLQLbFFwcMp0yTiFEGn4GbBAwGb5lZieyGNXaaYBh0XfBbEGznrberl5/z1hnEHSFuuHCJtZXk+Mjs0yp0jh8uiXity7hABIhP8COhjvo6kWQGsDcUDkdiQM9NAEcTaKkDSgz9w2KLQiIiPgvxaInblehyANReuCmC2TPjkaCvToAoYHaBRu7WI8n5FehrpcaoeyR4cJK3SmqeW575aWnMnW3+a/VALC4lPBHwk8a9Z+i90xAHoO832KNvcRrPL4p5IW5m2R8mgRmiOHSo7ikg59yoYiGp4+wagEy+SsPnXgIOiXQkN4KTEx0S8GUXsJpcfs7FvmGccIG3584nTVq7tUdLdPwZ8nVJ8FNJbgvdBJxDTzPdCmZktltZ5pQTGBQci2Jmk/1KmeJ+x/pGL7pBktsw4RNrK9gsaTPecbubm50Hxxwj2CvYBJ6l6CRwizWwxCZgGTgYcCOKtBG8063zAme4TZ2uRmq0AHs62la2atD7Nr48dL4soCkVBRIegI6rX/3qqAtL+QGVml6P1gbQETgsOQOyJyK81Kfazr+eUd6KxlcaLC2x1+E9/32T/6IwatUZKlJBC0BnQD3Th94KZXbqFdSAnAw4Ar0ZKfyjL4j3qPUP8zx7CtpXHPZG2OkQIaYJ/On20LDuKIqmO1BmReoVaq7W90MbMLlUDOBXBIUm7A3aXig/o6D7B9phtd+PMrgQP4dnqESE+WT/OzPThMvFhkN4VfAgchZjC8yPN7NJNCD4R/NdIxb+Wqr/PgAOkrWzuibTVpZqTNMbg6cOlOrtS1joF/YGCqvRPq7i0eyTN7Lu0hrCngCOB9ibFvzWL/Ec29ZzxKmxb6RwibXXauG6CoYkjOeudpIBQk+qCcCtQxyHSzL5bCZwJ4guF3o4cu5qp+ISN3Q6Qtio4RNrq9Ew0GdRpyvH9uV7OJBUh1BMwQLViuxUkHSbNbCEt+HMmiC8Ev4tSvy9z3se6vhMOkLZaOETa6rU9ZhnUKZpjOYW6s+gVqgVsRbGRoLvdTTSzJUfALDAOfC30VoReLYtiLx39J3nWcyBt9XCItNVte5S8oJHmuqmDtWauZ1SKqEfQRbU1okv/mNlCGRgJ+EzinYBXS+UPGO8fciFxW20cIs1+GQ1e1IlmfSrXEmTyRoiNVPtsd+PyP2ar3cKtDKeCOCTl1yP4fam8j/rAUQdIW41c4scM4LmYpqP7VDM1D0akd4l4F/gcGKW6cJjZ6tYExoAvhd6PFLvLXHuH2YGv2R5T7W6cWTt4qM6sZfBXJe9NNlXUZlOmuigEA1S72tQXHOkeSbPVYeEimrFAB4l4NcTvS+V3WN93lO0x084GmrWTh7PNWqpdbcb4p9Ofl3Q0ihQh5V4iOoAbqYa2/Z4xW11adSC/EvFuZHaWZfkOXQPHeNYB0lY3D2ebLdTa1abR/LqM2AfxBhG7qYa2J/CuNmarSaaa0nIQYk9Ir5WFPqAcOOohbDP3qph90/OR2aFRto4ezLloFjCuiBKpE+hlfjjbw9pmK1Prw2IT+Irgjcj6fZl5h3LyK55bM93OxpktFe6JNLuQ5yPDmhFoflESH5D1tuCPEF9BTFL1UJjZypPnviaAI6D3Q7G7LHgPTR7hbzZNtrl9ZkuGQ6TZxWyPkgMDI6g8nKX3g9gl9DboK4TnQpmtXJPAYcE7ROwqI95nYvoIf3vdBBGe0mI2x8NxZt9lp2ocHl9PXbenHA8r+JMQjxFcTzUlxO8js+WvVQtyluAImbeBVzP5TdAn/PXAsAOk2bk8J9LsuzwTTXbqNIfHP811FUkxoGB9VDvarMfFyM2Wu1Yh8QZiSBH7I/FWzuWb1PmUfQMjDpBm3+QQafZ9PBNNBjUMUwcjq48y1xWaDeJeYCNVHUmHSLPlp9UDOQ18DbwTsDuT36IZn1HvH6nmSJvZ+Twn0uz72h6zTHSfKjX7UYrYFRF7BJ8Aw3hXG7PlbAbipGAfiT9kytcpYj8H+8+wPfzeNrsI90Sa/RDPxTQv6lizYyQVUfQRrENaA9EJ9DE/tG1mS1+mKuNzWtKBSOzN6B2yDvAf+obZ7iFss2/jnkizH+rvmGG6MVQqDgR6B2Iv4iDz+2z7wmO2tLWGsBvACaE/RvBayrGLRnxKfWDUcyDNvptDpNkPFSG+3DBBTUfLUh8S7CXF+8DXEDPMX6DMbOlpvT9LYCLgUETsTcSbzXr6iM19pz2Ebfb9eDjb7FI8H5lBjdAxcSg3cy1BHdFBqAA2U+2zXbS5lWZ2YQ1gNKRDing70FvNaO6HtUM8E812N85suXCINLtU22OWF3SKnsnIKaeUCDIQ1ICbgC48P9JsKRFzu9EEcVDBmxG8WkZ+nzVrj/JszLa7gWbLiUOk2eX4ZTR4RacZn/oslWXKEXWh7oAeXPrHbCmZ284wZkBHhN4N4o2yWb5L58DXPBvehcrsB3KINLtcz8YMgxpqMqkaOWXokuiN4F7gWrxi26zdzvZAgr5Gei+C3SV6h4E1X/PvY6rdDTRbjhwizRbD9pjlFZ1qjk99WpSNbkjdglpUAfIa/F4za5cFAZIjiPcg3igV71CfPMJf9TtAml0ir842WyzPxgx0D5VZHwV6K6S9AZ9RXbyaeMW22dXWWoU9BnwFvAu8mklvUp/9nH0bJ13Kx+zSuXfEbDH9gmleHjhazoymoii6hfqBAaqFNr14aNvsamr1QB4G3ke8llPaRTFzkH3rxrydodnlcYg0W0xVr8YUvz52tEx9H6ZS/UAPESVwM7AGh0izq0FUIwDHgX2IXZn0JsXMQX6xbtS70ZhdPg9nm10Jf7NpklQezqnYG8HvidgLnGB+RxsXJDe7MlpzIJvAKREfQ+zOym9Rb37OvnVjHsI2WxwOkWZXQoTYt2aY6dlPS9KbSHuqixknEFNUYdLMFp+AWeCk4ECgt7K0lzI+gTUjHsI2WzweVjO7kgZV0BhZQ6StCZ4S8UTAVogbQT34PWi2mFo9kMclPopgTyJebUa8T9Fz3NsZmi0uz4k0u5K2R8mgRmmO7c/SbCImlUghDVBtjQgOkmaLoTWMPSxxIBJ/yOIPGfYz2nOaXzpAmi02h0izK60KkmeYGvss19WRFOsFA1Fd9NYCHXhqidmlas1vbAInAz4hYk/K7Ml1Pobekw6QZleGL1xmV8P2KJnuHyXXDiXizQj+IPE+MITnR5pdrhIYFXysiDdCeVeTYj/0nvYQttmV455Is6ul2md7qDk+9cdas5zNwczcrjYDVB/oEh7aNvshMvPFxA+H9FZE/kOp+nvc2HWMZ6LZ5vaZrWjuiTS7muZ2tWlG8UkovRPSe8AXwDjVBdHMvj8BI4EOEvF2ROwpyfvo6D7hAGl25TlEml1t22OWovtkmZsHIsXrwH8N+BAYxvUjzb7LwjqrJfCVFLsj619K5Xco1h5ne8y2t4lmq4OHs83a4RdM89LU0TJ3R0E0VBU/7kZ0E3RSbY9oZhfWBGaAEcS+iHitjMYuBtYerXr7zexqcIg0a4cqNE4wqCNlY5SC1C/yBiJ6IW4AdeE5kmbna5XxmQS+VsRnIb1RZr1Hx9rDPOtFNGZXk3s7zNpp8Fcl+0dnRG06kSYRTYIeoB+ozx3lIGl2boA8SPBqyvHPWbxBnviCv+53D6TZVeaeSLN2ihDSBP90+suy1lEWEZJyLyn6EDcCXVQXTwdJW+1KqgVoX4Hei5xeK8vm62wY+JpnHSDN2sEh0qzdqiA5xj+cPFJ2dXemiHUSPVH1ulwH9OJRA1vdBDED+pKIvSG9Vkrv0DVwzHMgzdrHIdJsKajmSI7x4plDuZ66UiQRzALbgM1UPZGupmCrTWsIuwQNg96PHDvL4E3W933uAGnWXg6RZkvJ7NoJeicO5YY6EuqptkeMfmA9niNpq4+AKSKGJD6MYFeZeI80fYS/7HcZH7M2c4g0W0p+GQ0GdZr61Gcp566c1SvRGcFWzg2SZitZq1ZqBo6StScSr+ccr1GfPcQvNk7M9d6bWRs5RJotNdtjlld0snly5JMipQ5SqkeoS6ITWNvu5pldBQJmqfbD3hfodznidfp7D/JXvVMOkGZLg0Ok2VL0l8zy8sCxMo+kQtEl0Qf0Q3SAOpnfa9tsJZqBOC7pQITeyBRvkyYO8Vd9DpBmS4hDpNlSVF0op/g/Tg2VtWJ/QXQLdSIJ+BHQ0+YWmi22hUPYZxTaG/BvORW7mJr9jAMbJ9nuAGm2lDhEmi1ln6wf586Rr8pUrxfkuqATogvYgne0sZWnBEYDPiXrzVwUr9Gc/pS/Wz/mHkizpcfDYWZL2fORqQ+MMlt+WUbxRxRvCT4CzgANql4bX1xtuctUr+fjgncV/C6l2M3U7EH+4zWjDpBmS5NDpNlStz1KDvafIZef5+CPId6L4DOqINlsd/PMFkGmWkTzQaD/N5f6/5oF++hdO9buhpnZxXk422w5eD4yOzXMsamDEc0Bib4gsuBOYAPe0caWJ1F9EBoL6XOIPTmxi8gfwsAo26NsdwPN7OIcIs2Wi2eiyYsaKouRPxb1NLfGhh6IdThE2vKUgROCD4h4PYk/5EZ8zGcDIzwfud2NM7Nv5xBptpw8F9O8qKMlIxS1oksR1yCtA64FuqjCpBfb2FInqkU0Y4JPAv1bDn6f4QCb+4f5WwdIs+XAvRdmy83/9auSL7tm1JxtJgREAXQCvXN/OkTaUtcEhiP4hIjXkuLVjPZR6z/Dv/MQttly4Z5Is+WmWqk6ya+PfV7WepqFYlLBDKgH6KZ6XztI2lKkua8xwQeIPyR4tRn8kev7h3nGAdJsOXGINFuu/va6CX6jL8vR8ZQS/VJsCtQLXIODpC09opoDOQUcCWJPglebZXzATT2neCZcacBsmXGJH7Pl7K+YotH8OpPeCtgpeAP4kmrf4Vavj1m7teZAngm0n2BXoDebkfZzU8+QA6TZ8uSeSLPlLELs0Chbz3ySG51KoXHELEEXVemfOu6RtPYTYjYiDgreCHi1LPN7dPaddIA0W77cE2m23D0fmX3rxqBxMCu/Q4rdgg+AIapdQNwbae3S6oGcITgl2BfBG2Uu93LNmsNsj6l2N9DMLp17KMxWihdUp39sgNBtibQN8STwGPAj5j8w+j1vV0trDuTpIA5KeieIV8tCe0l9hxwgzZY/D2ebrRS/jAY7dJo7R8qc0mxSzopYE+gaoJ/5kl4OknaltebjTiAOK7Qr0M6S/CGzU1/zt/0OkGYrgC8mZivNoAoaI2tQ3JVS/FzEnwVxO2gTVUFyv+/tShLVwq4zERxS5u2A35XU3mS26zjPxXS7G2hmi8MXE7OVaIcSW4bX0NVxa8rlo0T8HPE0wXXMj0AEPgfYxem8P7/Pca2vkwr2hLQrkd5uRtrPaPcxfhmNK9VYM7v6fAExW6kGVTAx3E9HfUsK/VTw5wF3A2uBbkQiSFw4JJx/bvDinNVGc//nEQKVVPMbW/Mc54+q/t1ENElRIk2KOBSwM0d+g0Z8SlffMNtj9qo/BzO7ohwizVayVo9kR8dtRSofltLtBNdKGgB1oqgRCKJJIERADiBBBCII5eqWVIWHuMRAqbnzTfU4AunsfWnBfcbccSIg5s5R+pbHjPj2288//CLt10XOh2fbe8735o8V5+/zfKH7+fagHsS5jx9x9vvVT+sCj6tvHBM6ty2Kb1bgaD3Wwt/DhZ57IKRMtbq6GdVWhRkpn/3ZSBmpQVUFoIHUIMV4oMNlFO+SmofYt2aY570XttlK5BBpttJJwUvHe+hacw1ZG2ql1is0ANFN5A4AcpoBlUQEUECuo7kV3UEmUtXTJIn0PULkhUJXJkiITD77Zyugnn9sEN8IQAuPDYIcMX/sBdq08HtJ3wyr5/9MIpHnwlvSucExOPexq9+Czt52fvsv9Bj5G48XZ4/NC/6+8LmfHyLnH/+bv7MqmJfn3hbFOcef8xgLf5cLHr/1eK3/I1RC0YhCTcoF/2cJNTMZqQSVFLUmWU2KPElKo/T1nuEvmZ3bptPMViCHSLPVYlAFXXQyPdpFWeui0eikiGp+ZKkm9Y4mBcH0bEERNYq5QNVQFRTqHU3KBQGlY+7vpXT22JbyW3oGC2WmlelCTCnTfYFjpyLoXhAimxINdPbYqQjqBLUIpufOY13fEm6bcz+38D4u9Jj1BefE2oJevca3PJ/Z827rOO93cf7tFzu2Pvf389txIY0LPNduiYZ0zuN1RJy934WmLvC91mPWIs7+vmu5+h2oLMmNal/rtZuq+59GTCA2zbVlfO7PM4ivKd37aLbyOUSarTY7lPg5iSESE6RvhACAifM2ItiEGEecOS+83Lng3wcWnE/u/JZAN7Tgtn2XOddy63nnsMu9v4v51VWYE/qrRTgfX6idl3O/rftzb6KZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmZmVmbRbsbYGZXiRT8iuDnpHY3ZcUbQt96+77vuH2xbD3vHL9xEc755z+3fYhfzX0v4uo8LzNbEhwizVY6KfhnOjg+3E1nvZNaqlGSKAga8e3ngLrmQ0H5A4LPwmOLi5xnzr+/ix13vtZxF3uMH9LOS3Gx+/8+behAlHO/0yaiU6K5iO2tEcxEUJtrSzH3/zvb+veCNn5Xe1u3NyLOvg5KdPbY1nMplJlWRipZ1z/FNDNsj3LRnpOZLVkOkWYr2aAKJob76a5dXyM2S2wg526lqJMJYu4cEMQF/p2BkjQXGrQgaLSOb0kLbmvdj9A53z//PqRzb4sLBFqRzrbp/PtJ6Oz9VY+Xzj5u62cTOqed51t4+8KfPf+283/m/Da2fr4lzz2XtOA55vOfn0qC/I3f68K/tx7//N/vQgt/RjHflur5fNs5fv7/u3qmC/5vzv5cLPhO9fsO5QXHzX2PHNJMpHSoSeMzptce47mY/pbHNrMVwCHSbKUaVAfNsTVEsblA90r5ARG3AGuBWuIbQSE4N1gIONujlJkPGem7zx3z4WOBfN6/F9x+wftLrTZ9kxbe34LjYsF9prm/f1tb528/P0S2bjsnpJ29fb7dFw9r3/47mg+Q597X/GNd7OfP/92eGzzP/nzEBY69WPtELAi8ioWvifnbzm3vQjmLmQi9F4pdZdb7dE5/xb6Nkzwf+QLHm9kK4BBpthLtVI2jE9dAviMR9wMPBXpQxBaqEHmxcGZ2qRrAfoLdIe0qU+whNQ6zb92Yg6TZyuSLiNlKs0OJLcNr6CruS8SfSvwk4GbgWqAXqM0d6fe/LaYMTAJfA+8Cv8uR3mR69lMOrR11kDRbeWrffYiZLSu3He+m1ntjytyriEcDPQx0Ax1UQ7wOj3YlJKoPKZsJEhkllc3cXW+wdegzYLzN7TOzReYQabZSSME/nOyjo+uWotQ2BY8H3Mn88LXZ1dBJ5oZqfmYolZlMd8mL+tSLbcxWFodIs5XiZbro7ryxyPGI4C+AB4BNeP6jXT2tRUrdwC1EtVq8yDFTdg2P86K8attsBXHRYbPlTgoG1U1j9KZCcZ/QkwSPA7cAPe1unq06rQ8tBdJ1wP0KPVXk2qPURm/iRXW1uX1mtkjcE2m23L1MF43RmwqKx4SeDOIhwXWgbtwLae0TQCfELUhJoc6iKDpLRnexU4d4JprtbqCZXR6HSLPl7AXVaUxfW5AeVOS/gHhE6HqgE480WPslUC9wC0QhNF3UYqIcmppip447SJotbw6RZsvVK+pkZOraInSPsp6AeBS4jep97QBpS0kXsBm0TZnJgsZseUR72akhB0mz5csh0mw5ekF1zkxfV9B8UMFPg9gm2AjU8RC2LT0BdEDcTpBE1Gt1NZpD4++wQ6dcQ9JseXKINFtuXlCdnskNtdDdmfg58KeCzVQ1+hy+DUwhAAAgAElEQVQgbSnrg7gTKbJ0DNIZ7jg9A4y2u2Fm9sM5RJotJy+ozrqpa2tN3ZPRY4IHA7YAfXgI25a2uddnNUdS8NNUlsqpo4fBsQ+h7zTbo/zWezCzJcUh0my5GFRBc2J9ram7MvxE6LEgbqPajca9j7ac9AY8GBG9KejKs2rSceYDpFEi1O7Gmdn34xBpthzsUGJiuJ96ujVHPKqInwXxY2A91fvYIdKWkxpwjaBbmZkU6VRudEzwT6cPMqgJ90iaLQ8OkWZLnRT8li5G65tT5PsVsS3gbhwgbflKc1+9AXco6ZmkqOVU64Qz+4GRNrfPzL4Hh0izpe6l4z3UejYXKT2sSE+GdC8wQLUS22w5CyKuCelhSX2JQrlRzPAbHeCvmPLQttnSVrS7AWb2LQbVTZlvKCI9oMSfIB4Dbqaqu+eFNLYS1IC+CNYKSDCrsjnDR1Mz3PO/zvC75x0kzZYo90SaLVWD6qCc2lDUuEvSEygeBW6lCpAewraVIBZ89QfxkBL1lMuB3OTfuOP0h7j8j9mS5RBpthS9qC6YuramfE+O9HjAtrlakF1UIwgOkbbS1ECbQnSJ6EqRpnKRxvn1sc/5m02THto2W3p8ITJbagbVAVPX1Zr5nhx6SuLpuYU01zC/I43ZSiSgBI4JdoX4XU55D1kHODAw4p1tzJYW90SaLSVVgNxYa+a7svSExFMR3EO1EtsLaWw1KBDrI3hYwUBS9OSi1mDr0AFgvN2NM7N5nphvtlTsVA0m1tca5Z2ZvE3BE3MBcmEPpHshbSWrXuNBF3BzwE8Ef1KofJDcdQMvqgvJ7wGzJcKrs82Wgp2qcXh8PeIuQk+J9HSIrQQbqUYM/IHPVpOzC26CCEQ9BaGiMcH+kUle/vvZdjfQzDycbdZ+gyoYGl9HkbekSA8o4vEFAbID9z7a6hRUH56uJ+KJLHoK1Cxnupq8qC94Lqbb3UCz1c4h0qydBlUwNbaWQrelSNuAp0LcB9oEdLa7eWZtFkAvsDkiaiKGi4LpMo1mXtRhB0mz9nKINGuX1n7YHfXNKelBxE+BbcCNeA6kGcy//juRNgKPCXIRBWXH9CyDOsr28NC2WZs4RJq1ww4ltgyvobN+cyG2itgGuhe4Hs+BNFuoFSS7qYrtI5SL1JguKd9gUENsj7J9zTNbvRwizdrh5pO9dHbdWMD9SnqKzEMEN1EVEzezb0pAD3BLoBmRhmsNDTdjrIl0ysXIza4+h0izq0kK/pFOukeuLUruUehJxJMEW6jmfnn42uzCWtM7+kRskfR4TkyhyPx67APgVHubZ7b6OESaXU2/pZtiZFOh+LFSPBJim+AOUBcewjb7PgJYE7A1RCRo5kIz/G9Ds3y5YcK72phdPQ6RZlfLK+rkzMh1RVE8qKwnI/So4BaqVdiu2Wr2/dWAAcG9IkZTpJHc2zPLHac/RRrz0LbZ1eEQaXY1DKqD8Yl1NWp35NBP53biuBXoxz2QZj9Ea8pHHVgT6MfAWCrL2Zw6m7w8dBBvj2h2VfjiZXalvaA6s1PX1hrcm5OeCHhIxI+o9sPuwO9Dsx+qNT+yAK4jeAD4SaH8MLl7M78+1tve5pmtDu6JNLuSdqrG0PjaWiN+nNGfSDwF3E61ytTMLl8PsJmIUDBeZCbLorvJizrM3zHjoW2zK8ch0uxK2aHE1yP9pPotOcqHRDwd8CDQh3sfzRZDAkQVJG8MxcNSniiImbJruME/rz0KzLS3iWYrl0Ok2ZWyZXgNqeO2RN4G8XBEbEFagxfRmC2moLqW9QndHimaUuQiitlyeGKKQZ10MXKzK8O9IWaLTQp+fayXro5bk/KTEv8N8CDSNbgOpNmVUgADEncKPaacHyWXtzIx3M8O+VpndgW4J9JsMe1Q4uUz/aTeGwuVD0rxRASPA+uoFtE4RJpdOQlYG3CnYCZFMZK7ayV3jnyKNOz5kWaLyyHSbDHdcbqPRuetRZEfluJJgnuYX4XdWlFqZldGq8dxY8DDCpFEPae6eJl9SNMOkmaLxyHSbLH8+lgvqWNzQd6mHH9OcD9wPS7jY3Y1pbmvdQH3CU2knEdzjE/w0sSXwESb22e2YjhEml0uKXiZLhqjNxSwVYpH5urWtXajce+j2dVXANeEuI9gvMhqlKmvxm/0Gf8+JtvdOLOVwCHS7HJIwUvHe+jouq5IxQMST0XEQ0Kb8BC2WTsFUIe4AXhcolYUOcqR0SkG9SXbY7bdDTRb7hwizS7HP9JJR9emIqf7JH5K8DOhVjFxD2GbtVeAeoDbCHqlNFOkfKpkaoqdOs4z0Wx3A82WM1/kzC7Vi+qia/i6QukeKR6J4EGqIexu/N4yWwpaIwE1YC3oIUX8WVGWT/L1yM28oHqb22e2rLkn0uxSvKA6XVMbi9xxj5KeFnoCuBPoxwHSbCnqArYAdYmOQmm67B8bRTrlFdtml8Zztcx+qJ2qcXh8fa1ga4Y/ieDngq3AWuZXhprZ0qK5rxLYB/w/WfxnSvbT1TfsOZJmP5wvdmY/xKAKhkbXUOQtOeIhRWwT/BgYoFoN6veU2dIUVO/PGnBjwE9TxH9Xq8XDlJMbvKuN2Q/n4Wyz72tQBYwO0OD2lNLDSI9GxD2I1n7Y7tk3Wx4GBPcEuSeTqJGnm3eOTLFDIzwfud2NM1suHCLNvq+xkz10d9+UyA8iPQ3xIFmbCG9naLbM1IC1Im4TOpOJ0xCTbD3zMdKo50iafT8OkWbfRQp+SzcjYzcUwb0iPQHahriZoAsPYZstJ60V2wnoDXSnxESKoplzrck/nPwEGGtvE82WB4dIs+/y0vEeOnpuLIq4V9JjwEPADXMB0j2QZstXDeL6gCQoknIjd3ZN8xsd4q+Yco+k2bdziDT7Nq+ok1Oj1xeK+yU9FSkel7gZFxM3WwkSUAc2BtqKGC8KTZWj4/DbvkOAt0c0+xYOkWYX84o6GZm6tki1OyU9FsHjErcDfThAmq0kHRCbCD2aFfUUquXxscxOfepdbcwuziHS7EJeUJ3TUxtrke/KoW2CbcAdwHq8H7bZSjL3flYP4pYQHUTMFMFIOTQ1xgs6wS+j0e5Gmi1F7k0xO98LqtM/sb4W5R0ZPSn0ZMBdwDocIM1WqgAKgk3Avcp6vMi6nzWTG9kpd7iYXYBDpNlCgypYN76WFDfnxH2KeDSIe4ENuBak2UrXmiN5Y0Q8ojI/VQvdy5HJjVWdWDNbyJ+uzFpaxcSbcXsiP4R4PEL3INYTfq+YrSL9gtsVNHOp2VpNs03Gm8BQuxtmtpT4wmgGsEOJieF+Ouo3J3hMoZ9EcB/EdYQ68TC22WoRQAdwTcDdCk3nzCxQ8r+PZrr7h9keZZvbaLYkeDjbTAq2DK+hu3ZTkbiP0FNBbANuAXVTvU8cIM1Wh9Z7vQ6sD9iqFE8mpccodAeMDnifbbOKeyJtdWvtRlOvby5y3iZ4CrgXtAFo9UCa2erSGnmoQ1wXxP1CzRRpJjdjktuOzwATbW6jWdv505StXq0AOTyyqQjuV6SfAT8DbgK62tw6M2u/aqFN1qaA+wieKsSD1Ho28xv1tLtxZu3mnkhbvV463kNH13VFSncrtA3xELCFahjLzAwg5rY4vQEoFRovIjXLybHMi/qSv2PG2yPaauUQaavTK+rk5Mi1hYqHlPRkwDbB9VTvCQ9hm1lL63zQA3Ez6AlJFDly2TU8yz+uPQZMt7OBZu3iEGmrjxS8PL6mVqvdnqWfQPyZYDPVfthmZudr7WrTBfEjkASzBcVk2TvVYKeOe3tEW43c42Kry6A6YGpjrcw/zllPKfh5wEPAGrwK28y+WwmMB/pExKsR+n1J7S2K7q9d+sdWG/dE2uoxqILmxLoixf1Z/DyCx4HbgV68yMzMvp8E9Iu4HRRSRJE0XTI+zaBOO0jaauIQaavDTtU4OraW4DbleCKkZxRxL1VRYfdAmtn31TpXrIG4M6SsnMdrYqrZGP+QQZ1ykLTVwiHSVj4p+D8nriG4M5G2CT1IxPVUdSDdA/n/s3enT3aVV77nv+vZ55zMk7NmgdAAQgMINM+AMfa91dXctjv6BYroW13uoiqi+DMU/Bl0RBXu8vWtiORV2xHy7apy44FJaEQjSAgkITQrp5OZJzPP3s+vX+w8TmGDMlNIyml9ImR4IRSPpa1nr72eZ63lnJuseh/JJpmtNahFsELRkpS+Y0DXFK/PuUfCg0g3u3Uq4Z2+dgjrg+xFwR7DNiDNxzz76ADQn/3T/uyfzn2XADQL1psgSjUo9NJ5c4R9i/unenHOPWyehXGzV6cSqpUOaoUng2yzsF2GNoEewyhP9fLctCGgBgyCVUf/PU7tktwMYUACdAjWCvYE4i6y5jV0drUj+YeIm9U8E+lmJ8n4eU8rjaUnA3E7kZ1mPINYSN5M3Dd3B3mwWCM/frwDAtGG0QaUyZ8V/9h29/KnO5JmPCeoBmISaw2BX9z4FB+P6GYxf5G62We/Ahu6W6mVngzoRRk/MthE3ky8Pg/bn/25S6M/IjAEdlvoMxOfY6TAYoPlQsvBFpMXXyVTuWA3I0QgBXqFDpnsNzHhd4wMfMnfLhn0qTZuNvJMpJtd9iuwtredtLQ8CTwvbJuh5xGPM1aJ7VwGdJtxXuIzMztlFi+CRoi2SGZPi7A5v/7AIsaeG//4cN8lkO8xHUZ4VqaBEKnFUmuJX3Z9DvRN8fqce+A8iHSzh2S8091K1rAqxGyH4AWwrcBS8gykc5BnjCoGnwn+zYifZAqXiIUupIyQthQUrkQ0bEar8mPtdjwb6SYmARaZ2XaJxhDVEmmo0anP2GcjU7045x4kDyLd7PFrygwXliaJNsjsB2AvgJaSZwc8g+REnoGsgL6UOBij/oMS56C590+9/TqVpPT3FGqWRMV5ZqFRaDVjgaQ/S+5eDNSEWGnQKpQkIdzOar0D/ErX+akNTvUCnXtQ/GjPzQ6dKlMZWpQkPCO0HXgWtAQPIF2uHkAOGHwK9kcz3ieGL1jU1veN5tD7LIOWrlThtMHvBb8z7HNgEK/aduOzu360GLZOij9OLLzMYP9TvK3GKV6fcw+MB5Fu5jugBmpDi5MkPivZNoPNwBN4EY3L1QtphoCvBB8b+n2Wxk9obOnhFUv/4r/YZxml8s2M4nGz+L7QceAaMEIeSHqRhJuIIuIxjB9E7NUk03Yaq4voVGmqF+bcg+BBpJvZ3lWBO4PzC2RrpLhNpq2C1UAbHkC6McPADcFZkx3NMp0ia792zztq+2yEkcYbWS2eNNkHoIPAF6O/Fngg6cYXMBrBHjfYKLPdSZZuIut53DOSbjbwINLNXJ1KuFxpx9KV0dgoscMsPAMsIO/v51wEhsFuCDtjZh9nwc4wMnKD121o3P/6dRsia7+WJTpkFv4Ds48wu57/mn/KcDr3beofsYH8Lu1i0HYF+6skFrbS2LOUt+T7lJvRvLDGzUz7Faj1tlEqPhlithFpmxkbEPW+fp6FnNvqAV4GdEnxrBnvRThIYpe4vHDiDaBftyF+pUtZpS9NSCyaFpgZRC3Js0z+nLlxGdAC9jTQrEAhiUmWNVdTOnWT16h5H0k3E3km0s08krHhVhNKHkvINmJhF8Ym4HFQGa+gnevqL+MIDJpx2UI4ERQ+QeEc3S09vGmTK5D5qQ3S2vZ1Zjpi8O9IH2FcIZ924y9/N576eMQWsOUm2yZ4OQnpdoZ6l/Fzb0HmZibPRLqZ59eUqZUWJ0FrJdsJ2kpeSONZIVc32sqHy4JjFnUkDelnJB3dvGG1+/oVf0KVX458FWn4YxJiVfmj1kzejLx+LOnPn/suo4GkyoLVmBWFdSRJaM6KPe/Tqa++0SXAuRnAg0g3s7ytRiq9S5OkuE4xbjFpq8xWAS1TvDI3vQwBV8GOmjiYJTpF6LjxvZo9mwmpwq+5kA32K2Q0yzTPZM8CCzHKD2z1bjYzoANoAjXLSBMrdWX0D9CpLg8k3Uzix9luZpCMt9VIc3VxYmGDFPdI7JbZk+QBZMCzQC4/wh7B7JbgjBE/yOAY6cjXvMb4hTTjMRM/tUHSkcuR+LFZ+K0Chwl2g/xo2/tIuvHUi20agMWGbZTinkLGBtLKPDrlk5HcjOGZSDczvEORYnVhoZatjsF2G7yM8SxjAaSb2+r3EvNCmqjzZjqaKRxGwxf5m/mVB1q4cH5+P6t6zsVyIQ1RgyI2GtZM3hlA+AeNu7f689EstM4gjZAUzEgrt48g9XuhjZsJPIh0099bKkJ1UcHi+oi2Kdp2jPoYOg8gHeSBWw3oEpw246NAOBKL2WWYP/DAX8hvWmS/+tjQdzGmlEISFkk0GawH5uN3JN34jPw5aZVYKxiMZt00lnt459bnQP8Ur8+5cfkL2E1vktHS20KaPR2DXpDZK2Y8A8yb6qW5aUXAgBlfmPgoyD5ILXwKbb0P7Y7ZmxZZ1NZHKVwwCx8bvC84Q/7y92NtN1EB6DBsDdKOxOIWYnk5v1LTVC/MufF4EOmmr04l/PPtFqywPMg2S7xssAtYijcTdzkBKXngdkWR40b4MA12kqR846EXKbxiKadb7mTikyD7ncGHwBXyynBv/+MmwoAC0mLqU21kW6j0LfNm5G668yDSTU+SMdDTSnPTmoS4E9MuE6vJW6oUyZ9df35dBPpBnwPvWeCPWRg5Q9J0+3tVYk/Gmxbpa7qVKnxqpo+BDw07B/SSB5GelXTjMxqAxxA7FLU3IWymrWeZz9l205nfiXTT0zu3mik3rQyZXorBfmBiE7CQvGGvc/WJNCnQBZwy8cdM8SjZ8C3+90cUQNa9YTUO6FZ2u/dkUkgapThoFqS8jUsjYxlJvyPpvk29YrsJWGkolVmaKKlltd7AAX3NqzY8zq/h3CPnQaSbfjpvtpA2PpEYG2Xaa9I2sMcxn0Tj/iQDGwZ1CTtr4nBmOk6h7Qp/016dkhW9asMc0NWsu/dwEpIolFdri6VAGc+cu3vLj7WhTWarDVJBmhCUdVVHeEs37rtRvnMPiQeRbvqQjN9Qoq9/eSLbKrQb2AC2gLFRhh5EOiFGMN2Q2RlT/ENEH1MYvsK+tqkJIOtetWE6dSUbrhSTAvOibIHlz+zj5H0B/fl191LPSLYJ1hqGAsMFst60tTKEdMdb/7jpxININ328QyPDfcuSYLtk+iHYNmAFY+MM/QU8t9WPsCNGj8zOWNT/G0k+pJh+yelFg1O9QABeo8Y7Q19nWfPhYLFEJMVsF+hx8vu8/hy7e6kPTmgTrFZkMJp1YbHKO32RTj28jgPOTZIHkW56+JWaGKw8kSThecEOYDN5ANmE34N0OQHDQMXgAlFHYoiHGEovcK6jjzdtehSw5OMRB/h5z5exVAoBFURsNawBWIxPV3Ljq/eQ7DBjnVBPiEkas8zIKufpVI8Hkm468CDSTb0DaqCv8kQSw06Z9lrUdpktJw8g/R6Zq0uBbsE5zA6a7GNqfElzW2XaBJB1ZmK/+ljVcyEWk8YQrFWiZPnzfHczcufupQAsNNgsUxIsKcSCZQz0nEfq9aNtN9U8iHRTq1MluqqLkoQNQi8jXpKFJ0Bl/AjbjR1h15uJXwY7aNH+kAU7SX9LNz+bphmZNy0i9fIvlU9DYo3RVJJIDJ4HOvD9143PgBKwwqAsKIYYq7GhOMw/3/6SvB+pc1PGszxu6nSqTK33iaSg54VtN9gALB1tieIBpKvLgD7gssRJizqexfQcSfk2/0g61Yu7JzMx1NqXEs4Z8aBhBw3Okf//yfBm5O7e6oU2CbBgdI98OZheorG8ml+pCcn3STdl/EvYTY1Olch6liQh2aQs22tmuwRP43cg3TcJGAK+BjtqxI+zxE6RVa/xWtvQjDjO+0dSflO+mfVzMsnSRNGazFQSthpow+9IuvEFoAG0BLMdkorBlMb+yhC/br0MTI+iMjfneBDpHr1OlRipLi4UimujtN2M3YJngHY8gHRjBGSIbgXOmXQ4U3KM6sgV/m7J4IwIIIHRdQ5zQDey7t5TiRWaJUqjfU/rU5g8kHQT0YD0mGEGcSCxMJj1V+BtXeR1G5rqxbm5x4NI92h1qkRWXVIIcX2M2iGxHeNp8jti/iJ1cHcrHxiwwFUiZ6LpBGl2keaOyowJIO+W95C8lsX+40lmQVAavbrxBJ6BdxMzekdSy8C2SdQSBcuaq4O8pWvejNw9ah5EukfnLRXJBhcWpA3RtFeRXWY8T16t6gGku1sGDACXhZ0200my5AJDTX38n9O0kGYi9lmVzptfZbXGmAQaZZZPshHLMe+H6sY19nyIlZgKCmbJSNqVtQ2mvKubvGLT+56wm1W8sMY9GvsVmNfXCtmqqLhJYqcZG8nnYRfwF6fL1TOMVeArzI6a4kdZ4AwLmrpmRabltUUDNLRezghHgPeAYwS7AYzghTZuYgyjEbEMaZtC2FuIep7rg4vplGe03SPjmUj38HUqoVrpIOPpEG2rAjst8gzGArxfnhtTP8ZOwbpAZ5E+yhQOkw5f5lUbnuoFPhD5UXyVt7svxlIpCcQgNM/yI+2F5L8H/lHl7sVG/7eRvP3PjyKxWFCwNK0c9vGI7lHxTKR7uDqVUOttoyF5KsSwA+MFE5swluJ3wNw3CTEE3JZ0AXQ8Kh6jOPIlfzN/9vXD+7uOXjR0PoZ4yOCwwRdAL/ldUOcmwoAWoY0GL0fTXkyr+XlPu2ck3aPgQaR7eCRjpKuZJFmWxOw5TLuAzWBL8SNsN6aegaxhXBc6bvBesPAxWbjI6Xkzs5BmPGbib+ZXkF0Isj/KeNfgDNCPB5JufPX7kQnQILMVitoelOyhGNZTrXR4D0n3sPlxtnt4fk2ZUFqeyDYKdgFbgWXeTNzdpR4cZsCg4AsT7wf4ME04S19L97QbafggmYlOdaf0Hy1klkYogOYBTwGN9Z81hSt0M4MBHWa2SagYLLTGUhS/uHGKvEDNuYfCM5Hu4fiVmhisPJFgGyXtNmwLY61MvBLbjRFV4JZh5w07GpJwOC3aWWi+NSsKacazzzJOt9xJUztjFj8CHQbqI+0iXmzjxlcfj7jQ4DkZLwTCdgotq+nsame//F3vHgrPRLoHSzLeoZFK79LEwrNCu8zYIWk1YwGkc3URo2JwXsRDBh+mWfEzQmMX+2ZwK5/JetMinbqTDfedTJKkTaZhxG6w1XfNkXduPEWgw8Qzgr4gWYwNB1lx+1Ok/ll5LcRNKQ8i3YP1G0oM9S5NCmGDCLsM9khaC7TgGUiXq7/IIlADbss4EWL49zSEM3Q03uRVG5nC9U2NfTbC27qSFXp+n5B0i1DEaEU8Th4c+AeYu5f63lrEWGLwkqA9RDXE5sZ+3uEieess5x4YDyLdg7Nfga7qoiQpPiOLuyXtBlaRB5AJHkC6MRn5C+0m4pSho6mS07SXb82aVj7343Ub4oCuZb09BOwxiXn5iDstARrwu8RufPW9dr7Bepn6k8xuZbEi/uX6V/xsqd+RdA+Mf9m6B6NTJVb3LywkcZ1C3C2x12AtY/Ow/cXn6gQ2DFwHjoB9mEmnKZXvzOkAsu5VG6a542ZUOGzit6APgWvAMGOV7M7dSyDPXi80sUmmvw5mP6TU8jSdKk314tzs4ZlI9/29pSLDgwtIWBMVtwp2GPYM+ThD71Xm7pa38kHdQufNwkfR4iFq4SL/laGpXty08VMb5O3uC7FUzBKLw1GhzVADsIA8I+nc+PJm5I8jXhAUE8XhLFS7OaAb/sHmHgTPRLrvZ78CrZV2LHsqmO0QtsssrGMsgPR7kK5O5MfYvWZcMsKJACdIwhcsb+nyS/9/5u86etHwF1lMjho6MtqMvA+v2HYTY+T7b8PoHcnnhO1K0nQTvYMLeFeeRHLfmweR7v51KuGpyjykp0Ji2wSvmNlWopaQZ7n9+XJ1oxlI7ghOC3vPiAdTahdoae7mFUuneoHTjpk4P78fyy5ZCB/lv2d8BgySB+MeSLqJqB9tLwNtF/bjJGY7uFVd4lNt3PflXyLu/ryrApcr7RS0OmCbkXYYbEIsxPy4zX3D6D0+q4Gumjhp6FAW4klChx+r3cubFnlXPdmt6okkZgXJmvOZ87YS1ITP2XYTY0Az2FOGJAvDhTQOpml/DemWnwK4++VBpJu8TiVcrrRTSp4MMe7E2A1sAhaRN7wFf7G5nMiPX0eQ7mCcN3Qii3aWlvZr/MTvQY7rFUt5Vzeyq72nk1BolWIHYGArRgNJ58ZT349blQeSAxEGCkVq6b9WTtGp7jnVl9U9MB5Euskb6GmlXHwiUbYxmv3IYDt5AFnEg0f3l2pAl+BTQ0eyJDnBwOBX/KSl6hmQCXrFUjp1JRuuHEkSGiQZgSJiOd5D0k1Mfc52m+A5QYwCZJGBnhNAz9Quz81EvvG4iduvQGdXO+XSU4niNsl2GzxLHkA24M+TG1NvRROBbsFZC3wQsMMEfUXrwkEPICdpn40wMnw1C3aIYB9IfAJ2DUjxYhs3MfVim1aDdZJ2h2A7aSw9zX+70zbVi3Mzj7/03cTsV2DF7WZGisuSLNsi8QrwAvA43nLE/SWRBzcDJl0x6Xgw+zDNwlm6m+/40dl9+vuF/YTqxUg4bNiHgtPATWDuTfhx98vITyEXmIVnhPYkZFsolFbQebNlqhfnZhY/znYTs/pGGWt6PEl4VthOxE5gBWMBpB9ju7tl5O1orsrCSTOOpQpneaJ8yyuxvwczIQ3w33svREuag2gSFM14HlhIfrTt3HgMKCItNtgg7E6CDWWxnNGpi7zGkJ8UuInwINKNr1Mlsp5FSbQNknYbbJXsCYwSHjy6bzcEXAEdRzqYxXCSmgeQD4SZ6FQfw4PnQqJSlEpmtEmUgTZ8NKK7N7vr3/Jm5GirIE0yhrKR3iF+036VfEKSc/fkx9nu3t5VgXRgXiEW18n4AdiPha0ZbePjz4+7W/0OZAr0C84jPohJOAjDl7joR64PzD7LGGy6ncZw1tFzj/gAACAASURBVEL8KL8fyVXEEGP3Iz2T5O6lfj+yDPYUYreMF5KiPUtvdTFvybPablyeiXTfrVMJ1yodGKsjbDJss2Al0IpnOtxfikAVqBhcQDoWSY6QDn3J38yv+PHYA/aG1ejUzaxmpxLLFkqxCbN6659G/CPPTUxCvqevNmNQCv0FxTQt9w+zX7d50+JUL9BNX77JuG8nGfS1g20IkZdktlfwFNCEH5e5b6pnvVKw25KOS/w2YB+SZhf5Pxb0eQD5kOyzEeY1Xs/QYTPeRToMuk7eVsmzkW4i6oU2LRJrDF6MppcoaC1re9vZL48T3HfyTKT7S5Lxz7dbaG5aFYgvyuw/mVhPPg+7gAeQbszdrXwGJF0w490Ywu/icPYF5VbvPfewvWrDdOqLbLiSJgUKUbRaXvBW790K/nfWjc+AeYLnDZIAvTEUh9jQfZ79qnhG0n0b/8Jw3yQZv+xqpVx+MlG2WbDdYC0eQLrvlpJXYl8y0ydB4RjVkfOUW3u8lc8jss+qpK2XMgsfm9kfhB0DbpH/2Xg20k1EPSPZJlgjbFeIcTexYS1rulry0ynnvskzkW5MPQPZ2LQmWNwt2R4znkMeQLrvJKDfpAsKdtCivZ+SnOdic59nLh6x122If7r1RWwsW0A1oYKZbSVv/ePcRCVAh8FOBWsIUjEWGkb4xY3PgYGpXpybXjyIdGN+2dVKuXF5QrZFsh8Am4k8PtoGwrm7ibwX5BBwTRZOmvggIztJsfGmB5BT5B8WVfiX6+djqTUExQ6J8uhUqXn4h6AbX/35aACeMKkgSBPUm5WaR/iX65f52VIPJN2f+HG2yzOQ/3SrlaRxdSLtEGEvsBl4fLSVj3N/TuSV2NcEZ0BHMjhNcfgarzE01Yub0/52ySDVkS+C+MCM3wlOAN3kQb9zk7HAYKPED5KoFyg0r+JteVLB/YkHkXOdZLxzq5nmxpUhxheE/TVoL+IJoIw/I+4vibz696bglGEfxRiPMly7zGuLBrwSe4qZiYsdfelwejQS/4dJ/2HwGVAhDyT9z8dNRP2O5DKwH0ez/yWRdtDYs9R7SLo6P86eyyTj15SpNTyWGM/KtANsC9gyzPvMub9wd8uYIcGXBkeC6WgshQssa+v3AHKayK8T9PB292exVGgI0hKwZozVQAv5320/2nbjCeRH24tN2izsWpIlt7PmakqnbrLPfIDAHOdBwlz2a8r0VpYnFjYK7QR7HngMVMZfMO4v1TOQfcDXBqfM4vE0Sc6zqKXbRxpOQxc7+ojhvMHvMf3O4DTQg2ck3eQkYIuBnUrsf0pCtpesusQzks4zkXORZPycBiq9S5MkeU7SC2a2e3QajTcTd99F5EeiV4DThh3JTJ8yUL7DPg8gp6U3LdKpO9lI9UiBOBSNDKkRs3X+segmIc9IGmsRZUFjIcbedN5Ajbd0hzesNtULdFPDM5Fz0c9poLm6oEBhjcx2jAaQzwJt+DGX+0v1Y+wMuIN00tDvMnSMkaHrvG5eSDOd7bMRBso30mgnDB3GOAvqxqfauImrJxbKwDJge0R7CpnWM29ggU+1mbv8D36u6VSZxp6lSdTGiPZI2il4Gmgm7w/mAaS7W30azTBwR9jnBDucxXCQQu0KF5ZUp3h9biLesBoNTXeywBmMD2V8DHwNXknvJszIY4YmYKWZ/XWM/G9JFvewbmCRH23PTR5EziVvqQjV+YVYXCfFPTJeMFgPdOABpPt2QgwDtw0uGJwI4hSlwS85Pc9Hoc0k+2yEUPsqxnjIZO8LnQN68fuRbnISoE3oWYyXJXsZtJ55/R10KpnqxblHy4PIueItFZk3sIBatjpa3KrIXoPngAXkm4Jzfy4/wjZ6JD6T9JFZPJSqcIHXFg14ADkDvTavjyKfR4uHDDtixgXyQNKDSDcR9WPtekZylcG2AC8WMjYw1O9H23OMF9bMBe+qwLVKOxlrQgg7he01tBloBepHEJ6FdHeLjDYUN+lrLBwJ2Lsp4TTFxhveymeGMhP71cOGvtNWS1qlrMGCJRKbyFu5eFGdm4h6INkiWGeGRVkLxZix4naVvADPzQH+xTDbdSrhWqUD0+qgsENoj6H15EU09TFo/tJwfy4jbwVzURZOGRxPrfYpQ+Vb3htuhnvTIt1tlYxw2gIfCA4DF8nnInuhjZuoeiDZLrFW0o4gdtPctIbOmy2ekZwb/A95NtuvQLXSQUieDDFslekFg42IpfgdSPftRoMIGwFugI6bxQ+zoJM0d9zk7xie6gW6B+ANqzGv8XoWdcqwj4HjwFXEEB5EuskpAvPMWC+zF0Kml8ia17Omq2WqF+YePg8iZ6v9CmzobqUhWZko24xpl4lNiGV4L0j37eqV2DVQn7ALmH2cxXCIUL3IT6j6MfYs8qoNk1WvZuIT4EPBGcxuAikeSLqJqb9HEmCRia0y/ShILxJKy33O9uznQeRstF+BFbebicXlSZZtEWEv2DZgMX7vyX23eiX2HYMvTToaIycpZj4Te7b62yWDFLIrMcZDhj6S/lSx7UVTbjKMPCO5xNAmob0JtoWk9zFv/TO7eRA520iWZyDLqxLZFll4AWk7sIK8Uaw3E3ffJs9CGl0SJ2X8NmAfouQLFrX1eQA5S5mJRW19lDgXsYMWOAy6DIwwVlzlf/ZuIkYDSVtssFmml5JispP23hUeSM5eHkTONr+40cRIcVmSsFloD3kV9grGmok7920iUDXjigU7FmQfpAqnmV++5TOxZ7lXLOV0Ww/Yp4YOgx0DviIvtMmmeHVu5mkAHkPaJngxUdhIs8/Znq08iJwtJKNTJZLGJUnCs4rahbSTfB52C15I476dyMff9QNXJX1mkZOpkrMMlG/wqnkhzVzwpkUKzbezGD8x7PeYfQB8CQzi2Ug3cfWK7TLYapPtVGRvUtDzzKsu9kBy9vEgcrb4DSWoLk2S4jrJthnaCfY0eSsfDx7dvQwDXyOOIzucJXaWYuUO/4hnIOeSfZZRbL+SFcIfTPH/QXxAnpEcZuxo27mJMKBZ6GkL9qKifpik6VbmVRf7VJvZxYPI2eCAGuivLimkcZ2ybIsUN8nsSfIMZL0XpHN3q1diR6BfZl9gOhqDPiFmlzm9aNDvQc5B+2wEytczKxyxYH8EOwhcHC248oykm6hA/u5plXjK0G4R9hbS+DzDgx5IziI+sWame0tF7gzOLyTx6Wi8JNkuM54hz0D6R4L7NvUAcgQYMOMi6ERQOBrNLlBp6vWRhnPYPhuhUzeyWDmcEJCIGIZYgVGe6uW5GSUBOgTrDFnESAIhy6oneVfX/L71zOdB5EzWqQT6OxiJKyPJRkXtNWMjMB+vwnbfrp5NSoEew74SOmbiaBrCZyTl27xhtSleo5tqeSB5MYu9WRILJtSKqQy2FCiN/izfX9y91FvJFYEFgmfNFGRYEiG7NjBCp26zz7x4awbzTNVM9a4KpJV5jMTVIQnbQNsN1gAdeADpvt3dAWSfGZdk+thkH2SRsyTl2z7S0P3Ja9QI7Vcz03GwD4V9grhBnsH2Y203UfVim1aJtSbtFXEPaD1pZZ4fbc9snomcifYrcLGnhXJpRZB2Cb1i8BzGIvKvPue+Sw2oGPalpEMGv8+y7BPa267yU6tO9eLcNJLfia3yT7cux8by7wMMyagZ7ASWMLbX+AerG0/9jmSHYK1gJKC+CCNUK+fYr26/QjMzeRA509THGcbSyiTqeZl2WWQLxmLGjpmcu1s9axSBAdAlwTGTHcySeJKOtq+8lY/7Tv+wqMI/3boQm8sKolXSPMsHF7Qz1nvWA0k3nvoJWYcZa2XWG2A4NoSUDd3nkHyowQzkx9kziWSs6WphpLQiUdwm017yDOQS8gDSN3L3bUYLaWwIuArhhGEfZok+IQxd9QDSjevvF/ZTrV0K6LAZ7wtOALfIM9vOTVR+R1IsMLFJ4sWQxe2MlFbway/amok8EzlTSMavKaPiY0nCJolXgK3AE+THSj4P232XCAyBbgJnQAezwBFC9SL7FvdP9eLcDGAmOlVJ04GzWMyChT5FpWZsAhbgwwzcxBlGI/C4mQWkWpLQn1X6hujUFfb5tZqZxDORM8VvKNHTuyQJrBfaDraVsXGGHkC67yZGMK4LO4t0NMb4CaH2Fa8tGpjqpbkZZJ9lFJpvk+h0VPzI0AngOpg3I3eTUS+0aUBahrFR0u4khE1k1YVeaDOzeCZyJuhUib7eJ5Ik2SFjtxnbJR4jv5fkf+Hctxl7oRsDkn1mxPci9iEluwBt/X7/yE3aPsvoVC+F6peWZocE7SABqxgbbuDcRATyU7TlwB5BISEdybLBjHd103tIzgyeiZzuOlUiG1yYyNbL+CHwssR68g3b//zcd8lnYosqcMfgRDD7iIxPOd1yx3uzufu2zzIGyneyoONm9kewY8A18DGZbtIMaAXWIPZKYU8hxnVcG1jAfvn7bQbwr8bp7IAa6K0uLpieiVnYAezAbA2oAQ8g3b1lQDfGTeCUmY6mSXKepeUufuatNNz39LoNcUCXsq7+QjCVZdZmeTCwEGgY/Vl+xcZNhJGfqq0U2h5hsCCS9KnKMTrV4x+805sHkdNVpxL6qwsTpZujeFlm2wxWgMr4HUj37cZmG4uaBT4X/N4svJfJTkH5lh8RuQfmVRvmbX0RywPFkFGWKTHYBizF3y1u4oz8WlazwQ5gfjTNp2g16DsJdE3t8ty9eDZrOnpLRYYHFxeyuF6ynYbtMVhP/qXvAaS7lwgMYnwtOGHoYBZ0jKR8w6fRuAfudRuiWrsUQ/aRYX+UdAq4Q3607Xdu3WQEoA3jaYOdQdpNauvo7J2P5O+8acq/Fqeb0TuQhahno+IOwTawtUAbPo3G3ZuAAeAK4hNDhzLpLEnzLT8Scg/NxY4+NvSdjyOxEMzmWT6VpBEvtHH3p0XwtLAfBSjEjAI/7zkJ9Ez1wtxf8kzkdPKuCjAwvxDjuoheFLxoxrPAPPLN2LOQ7rtEIMW4BTpmQb/Ngh0iq171ANI9VG9ahLZeMj41dHi0EfnXo61/Mjwj6Sau/o6bb8Yeg/8SZD+gVFzFr9TkGcnpx4PI6WK/Al/1z2ckro5mWwV7DNsItoT8z8n/rNy3GZ1GQwQqilzCwicZyTFC7Sv+dsngFK/PzQX7LGOotS+LfGroIOgY6MbolKSxu7rOje9PdyQF64XtSoibGaw84VNtph8PTKaDt1Rkdf/CQoHnQmCvUP0OZL3S0b++3L3UgC7DPjfjpEU7QxKvwDzvBekenTesRsPQ1xnhCGbvyzgO8TZj9yP9WXQTVW9I3mroWWH/OREv09//JL9S01Qvzo3xIHKqvasCrZV2ivHJKO0W/NBgM8ZixlplOHcvw8BVoRNmdjRLss9paenzY2z3yO1b3E+1+mWM4ZBhH0r2GWOFNs5NVgBbirFD0X6cyDbT07uEt+T1AdOEX3qeSu+qwOVKO6VkVUK2MZrtGm2R4bNo3Xi+eYwNFwyOZ5GTZINf82rb8BSvz81V/7Cown+7cz5mxcaQWLNE0WADY3e7wfc2NzEGagKewAiSbidJ6M/mVTMO6Aavmu9zU8yDyKmyX4FrlQ4SPR2I2yR2jh5hd5BniH2TdfciYASoGHwJHIvoBMV4mXNLqlO8NjfXnZ/fz5quk1FFC1gqwGAL35y05XucG4+R73WlPCOpl2ShOaml87Lu3j/wtq7yug1N9SLnMg8ip8J+BVbcbiaUlyUWn1dkL2ZbgceAEr65uu9Wv1cWgW6DixKfmOkEsXARyr15taxzUyh/Bvv4v/vOhgLFaNZsii0ye5q8361/KLuJGi20USOwCqkkwxILvVlpKOOArntGcup4EPmoScYvu1oIDY8n0nphWzA2IVZgfgfSTUhKnoH8QtIhC3Yws3iGQtlnYrvppdzak6YDZwqBYozBzDQisZ78aNu5yQhAM9hyoKZIb0JayLp7j/CWvuINq031AuciDyIftXdopJAsSjLWyWyLSc8LlmM04n0g3b2JPIDsB67KOGnY+xnpEXo7vvZN1E07+yxjv26lT1WOULChgCVIZYxmvnnq4vueG89oxbaagFUYPxA0JRbSrHVggE7d9o/oR8+rsx+lA2qgNrQ4yYrrFGybpM2CVeABpBtXvUVKP3AJOGnS0SxylqGO6x5AumnrTYusaO2lyFkzfYzZcfJnuJ+8GblzE1V/RzYBTwG7FHmpIJ6HgUVetf3oeRD5qHSqRPfQ0gLZeuWFNHsMniXvBVmfRuPcn9NdP1LgOmZHTPptFsIhhoe+4u/w+0BuenvFUha1dGe17IiZfgP8m6Fz5IGk9490k/GnZuTASjPbHY2/SkbiduYNLKBTyRSvb07x4+xH4YAa6K0uLiRxXcziNpltHa3EXkw+D9sDSHcvGTAE9Av7wtDhLLGDDFQv8fcLvaG4mxlesZROfZ1l1TQhHRZWAtrIT2LqR9u+F7qJqk+1WWcQFAiFVIMplRNId3xffDQ8E/mwdapE7+CCgrI1McZtCrZzNAO5gDyAdO7exAjGdeVH2IdjppOkI5e5vHDAN0o3o+yzEfrK1zPLTo7O2T6DuEneMN+fZTdZAWgXWmfYnggvYjzDO33z2C+Pbx4Bz0Q+TG+pyPDggkIhro3YFmF7TNqMWDBaie1f3e67jLXyMSoSpwz7Y7T4MUU7B/MHvJWPm5HesBr/cv1WlrQeC6YWmTCxFWMJPqXLTU79aLtFsEYwFKKFgCXp2t5PkHr8Q/vh8iDyYZGMf620E3kqYhuRtpqNHmGbH2G7CakfY980OB2JH1PjLIWWHq9CdDPaz5YO0Nn1ecwKDSFaqxkLlbf9KY3+DN8f3WQEoMNgoxlJhJSkOMA7nAJ8+MJD5EHkw7BfgXf6OgiFJwNxE9IOzDYBS/Bm4u7e6l/NGdBt6JKwYyaOosJFymUPIN3scHpehQ395y2GDhEXIJowniKfauNjX91khNEf8wRrJesLirdi2j/Ar3SRn9rgVC9wtvIg8kHbr8Da3nZicVWi7HkRtgFbwFaBylO7ODdDCBhCfCWzDw3eyxKdIngzcTeLvGmRTnVlxeqppEaj8vdRAjxJPtXGuckyYL6hdZjdTAIDWW8l44Au+lSbh8Mvnj5IkrGqp41QXJUo2yzZToibgOWjI5t81JcbTwYMYFzHdMawI1mmT2hqvcI+82MZN7vsswzK17OQHjV4D3QcuE5eaON3ft1kGXnB6mKitsi0N0nsOXp7HuOA/L7tQ+CZyAdFMt7pbqOhtCIh2yLZj4CtYE+QN0b13lVuPAIGgYtIJ5B9kCV2kurwVX7S6gGkm5322QgHdC3r7juSYC0RmsysiPQ4fv3HTU69TVQLxlOIIGkosTCU9VYzOnXVT3MeLA8iHwTJ+HlPOw2lFYnxnLAtwEZgOXlDVN8E3b3Um4lHUDfYGWQHo+Ixoi7TunDQKwzdrPaqDdN581pWaz4WULNQ0TADLWWsYtv3UTdRdzcj3yrZYEExTTUwwru6wyuWTvUCZwsPIr8vyfjn2y00lJcnxiZFbTdjs2AZeQbSufGI/PhuwLBLkk5Gs5Nk4QuWtfTwin85uzng9KJBNvR9GUesMRgNkhrMrJG8p66/q9xkFYDWvNCGNJrybhdf9Z9lv257i7QHw/9ifl+/uNFEqenxJOgZRXYBewQryS+G+wQGN576OMNug0sSx02coJh8TqXc7V/Mbs540yL71cOG/nOxRhICLSZbINRAvp96xbabqPpzEoBFZmwUhCANxwIjbOgeRurzE57vz4PI+yUZv7jRhJUfSwqsk9hogc0Sa8jHePkdSHcv9SNsgGFDVyX7xLCPM4ufQbmLN6w2lQt07pEbrdgmDp43y9oFHaAEbDXe+sdNXr2Ytd1gnUIYCGIoZg01fnHjLDAwxeub8TyIvF//FwXmtS1IRtL1UtgjtB3xJPlG55ucm4hI3gj3puC0oYOZcZzC8DVeY2iqF+fclNhnGZ26k2U9J5NYKCpYAdEIPMVYRxHfY91E1Su2F5i0SZgCUiy11PiVznsPye/Hg8j7sV+BxkobIyxXYptNegF4mnzignPjGS2iYQi4JfSZEY5niZ0gDFzitUU+E9vNbftshE5dzZJqTNLYJNQBtJEPbCiS/x3yQNJNRgl4zJAJaonUm1X6BumUt0/7HrxP5GTtV2DF7WZKyYpgbJDYJHgeWMzYyC7n7iW/Bym6JD437FhAJ4jZZfYt7vcA0jnyQJLy11nUUeBDodNAF+DXPNxk1esTEmCxieck252EsJVa72N0yt/d98mDyMnoVMJTlXk0lp4J6Mcy+yvLA8h6H0hvJu7upZ6BTIEuwacG7wXZ79IsnOWx1p4pXp9z08s+yyhVL0fs94b9WsYHoGvkgaRX17rJqh9tPw78QOJ/TSy8TKys4i0Vp3htM5IfZ0/UfgUGelppLK0IxK1SfMWw58mPsL2Ixo2nnl0cAXoMu4zFkxaSw2nQKbqbvHeZc99m3+J+Om9+EWOThUhJWIOhErAQ74Dh7k8TsAKsQWaDSab+bF51iLd0zQsaJ8czkRO1qqeNxtKTibJtyHaahWeAx8gfRv99dPdyVzNxegWfSjposoPZSO1TFjXf9o3LuXt4bdEA6fCXUdkHpvhHSceBG+QfZc5NlgGNoMeI2iJ4OcmyPbT1LPOj7cnxTOREHFADXf2PJzHbKuMl0BbEArzdhLu3u9v4pGA10HUzO2gWP8hq8SQL2696BtK5cZiJ/epnG6fi4KBCVE1EGbaH/HjSM5Juou5+TooYTwMNkloSFWpZUq3Rqes+HnFiPIgcz6/UxO3eJUmSbJDZbmAn+TSaxilemZsZ6s3Ee0C9gnNBOpqFeJqs/Rqv2vBUL9C5GSGfMDLI293nY0NSCGatiJXkQWQDY/fSnZuMMrASzGR2o5DGSloYyNivmz7VZnz+5XYvb6uRUt/yhLBZxl6TXpCFZ0Fl/MvXja8eQPYbnJY4Zehwhv5Aa9vX3p/MufsgGf9aWVCQPR/N/lpop8Fa8vY/XtzoJuPuThgVgxPCPjDZ77MsHqax5Y5nJO/NM5HfpVMlakNLkhA2SfY/G/F5ma0CNeIBpJu4DKxX6LTB/5eh44y0fcVP8Aykc/fDTLyrnvRa5SRoKJhVTARh7aP7sxc6uom6+z1eFqxHapIBBXUz0DOC1Ott176bp/6/TadKUF1UIFuvqF2gl0V4BujAv3TdxIi8mfgdoS+B05lxhmLbFV63Id+UnPseXrGUz1q7iLXTwexjwSegS0A/efbf/365ySoA8zBbJ7QjyLZRLj3FO91tSP7O/w6eifxzB9RAb3VxkqWbYrDdwrYaLPQjbDdBY5XYZtclHTf0UYTDKF71cYbOPSBvWmS/+tO1Q58llrbKMKQ9YE+TT7dxbjJGm5Gr0cRqBf5TiFk50pDwTvc5oHeqFzgdeRB5twNqoKu6qGBxXQz247zyTyvJL956AOnGUw8gU6Bf0gWTDkbxPiX7HNr7PAPp3AP0pkXe1o2s3Hs8UWKy0JJ3zlALPmfb3R/DWGRiq0QpZLEa09IQnfqc1/BTpD/jQWRdp0p0VRcVQrY2oq2GbRNaD9Q3I9+I3ESMAF1mfInsaDD7JBqfs6i5m1f8grZzD9zrNsTb+jorV5IglkhaYmYNwHzG2v84N1GBPHG01MwAKiEojdlggXeGv2C/Kl61PcbvRAK8pSLDgwsKlq2JaBvYNmHLgWY8gHQTJ6Bf8Jmw94z4YZqmn1Fp7vJekM49RK/bECODX0fiYTP+IDgBdJOfCjh3PxLyqUjbhP2XoOyvyRqeYvWN8lQvbDrxTOS7KnCrv4NMT0aFrYK9Bs+CFuBBtpuY+jSaGnDH4JQRD2YhnqSx/To/82k0zj10P1s6QGfXpzEtFIPRYKJJsAFox5MBbvIMKAHLTGoT1pSQ3c6Scj9v6yteN7/fzlwPkvYrcLnSTqqng2wb0g7LN53H8XuQbuIiUAEuCU4YOpalnKGp/ZoX0jj3CJ2eVwE+DbIPhA4Bl4EqXq3t7k8gDyTnGayR7OUk2A8pDqzhbfnAEeZyJrJTCfS1M8KaYMkLknabsQFYSh5Aeq8xN54ICGwIdENwyrCDWQiniEPX+AlVv4Tt3CP0pkU61Z1m1U8TYotEG/m9yBXk15M8MeAmyu76Z0AswNgesXIICrFQGeFtXZrrGcm5GUS+pSL0tVIrPBksbhe8bNhzwKLRZrVzO0PrJiI/whYjmLqEfW6KR0PgSIzxS0rzBzyAdG4K7LOMt3UrS3qPJ4VCs4hFsASxAqMBP9p298NoBFZYpIwxnCRhICtWxNu6PJcDybkXRHYqoVppIxZWJiHbJMK2/AhbS8nT1v6l6iaqhtEtuGCmo4FwKE3tM55o6vFKbOem0Os2xAFdze5UDiUJjRGazFQGW8LYPu/cZBjQgLEU2CJjOIkhyUpDw7ylq7wxN+++z62MW/0Iu5CsSrJsi6LtMMXnkBaQbyz+herGo7t+VAwumHTY0OFUyTkamu54JbZz08CrNszI8NXMkiOGfSBxBrjD2EQbPylwk5Efa+fXI55A2iXijxLSnbQMPZ7HF3PP3MlE7leAvnZqyYrEso3CdhnaImwVY0U0zo2nXoldRVwRHDfs48yy0xQbb7LPRqZ6gc65UZcXDrCq58vYmJQD1iwoWp4wWIDv+e7+GNCEWIlZVKAvoTacoZROXWff3DqFmhtBpGT8squFWsOKJMStku1GbJPZk0Arcy0j676PPICEm8CnFuxwpniS0H6VfVad4rU55+72pkWkXv61ci5C3vonql15M3LvA+zul43ekVxu0m5haZKlZMnAx3Tq1lwKJOdGEPkbShSKS5PIcyLsBe3EWMlYtZ4HkW489aOvGnBL4oKhExnhFOngV/zXNg8gnZuOzIR0h//eeyKEQkM0FgBNwFNAQ/1nTd0C3QxUr51okexZjJKiIfnT8QAAFSVJREFUhQIMpLH3KFLPXCmsnP3BU6dK9PY8lmQ8I+IWi3EjsIx8E0mYC78H7vu6eyZ2l+CYmf1bNN4nSS9xYYkHkM5NZ2ai2N6X1uykof8B/DvoDDCA341096+ekVxpxo4o/aBAspl/HVicX6Gb/WZ3JvKAGuitLi5YcW0Mel6R5zBWkR9hz8lLsG7S7g4gK4Z9jel4zOJHBD7jdFuPz1F1bgbYZxnv6lZ2a+BQkkYka8RoA5bjGUl3f+rPS7NgtWAgBiJRGRv6akDXVC7uUZi9QeRbKtJVXVQIcX2U/v/27uy5jvLM4/j3ebu1S5YXzGpwwCwxuwEHSEISV2pmKqmCOzwXcxPfTP4Mhz8jF5OkwiRTJe5IVeZqytkgGIMNGAzGQMALRsiWrX3rfp+56G6do+MjIyngoyP/Pi5ZUp9e3m6d7vfp5+3zvnvdeMKMXcBmNvJ+y9dtMQNp7mfcOGbub9MZPua9vlEFkCJtZJ9lHPQv83tn30lCPuDRN2E4cAu1x5tEVqN6JG6TwW6PxBBsKi6kU7zssxt90ImNGUwd8pSRqW0s+F0R3+PwfXN7iGIw9RRdKGTlHJgx+MThVTN/NU/St7nUc1EBpEgbesEih/xc/vnsKwlZdLMZ3J+meEay6upNZDWqcbZvMOMhd59IQhzNJ8Yu8ZvBz2HjDn+78U6WIU84O72dPN4VAg+BPWrYvRRdOnS0unjSNqqufBaACw4nLNjhfCG+BT0j12vHsiIbwj7LmO8ezpP8iDl/Aj8KfA7MUZz3IqtR34fkFoPd7vZUYske+ma2ccg3ZsKOjRZEHvKUbGILSf6tgO3BfS9QDmeo0Whk1aYwzrrxLnAsNz4gHzzP8xv3rlLkunHAZpkZPJfnsQwkOQycxnV+y5pVGcnbgCfc+FES/WHOT23bqB+02TjR8UEPjIxvwpMdSYgPevQnMXuM4qHpbhRAysqVWUgfdbfjhv89Rt5mgfMcsFkOtLp4IvK1OGCzDPk/8mzCEnP3aKH8sE1vq4smbamKMQaAXTi4xzHwCR64NAuMtbBs34iNExl/6/ImYnJ7ErjfsT2Y7cG5k+Jh6Y0TLMs3rWrCnjDsjDnHgoWjePIJdwxsuAuAyHVvv83TP3A6N3sds9cMOwOMA/PAddNptHxtqoRVH3CnO3sD4TvkXfcw5D24b6hkVvsHV+7GH+hhcvK2JOcRx582sz2O31L237RxAmX5JlWfnsuACwafOrxh+DsZ2cds2zyqMbFFNqjnbJpfXzobu8Pfg4dNjo8a4dvgt6JRbWT1jKIbwT4zHnI8BI9dMZ/OeWnqFDDZ4vJ9bdo7iHQ3XqKbhbGbE0u+7cVINM84fjNFOlknvqxGBKbNOO1w1NxezxN/n/m5EX6CxsQW2cj6Nk/A+Kk4Hy0Em3U8MdhCEUS2d10prZIANxg84s5C8Pxi9O4Zfu2fcsA2xLO37X1i/IEeZsduSjrCg+72uAV/1J27qXUcK7ISsfyaBb50t/fN45t5wnF6B87y7wMbup8vEYFyvONR/vviiWCd3TH6dsxuAXZQeyxKiQlZjVB+bTMLuyFeTDzM5umEM+Sn2W9tP9pZ+zb1/tG7mLp8Y5KGRzzajw3/njs7affAWFrE5oBhN94HP5aH5F3y/BzP2bQCSJHryKmtkxnJx2b2GvB34FNgGg2PKGtn4Ddg9pTjP00C3ydO3M4vve27HWzPgOuQp3wxvS31jnui+V7gGYc7gR50pygr43XfM9wvOv6hYUcjvEMSP+ODQX2QRuR684JFhnw4j2NvJXna7cRuzAYo6pcOVMfI2vTg7MDoclhIchvPt0yN80u/2M79DrdfJrIIIG9MzR+M5k+6+2PAHcAman1BiqxEpOhceMTNT1iwwwF7HQunuNSvMbFFrlf7bZ4w+Hke4xsEO+zYCWAEdUQuaxcoYpTtYI96YF+y4M/QM7aDIe9sdeHWqr0ykQc9cHpiME3tvgjPuPE9c9uF0Yv6gZTVcYqufC45nDK3NwJ2OEvDcS5pRBqR695+m+FlPxMnxo8Es02O9VgRBGymVneqzpGVqH+fpMAOnGfcvDdJw0LO1Azuw+346FT7BJFDnsD4IAvsihYec/y7Bo+VAWTS6uJJW3GK/t/GDD4Cf9OMN7MsP0k6qABSRArPMsNv7eOQWl8M9JjT7XAfSwNJkZWquv4ZwLkLs07HvkwzH89enIwc8rbrSq59ToKFsU1gdweSp8u+IO/EXR2Jy2p5+TUDftbhiBmv5DG+Qz54nv2mrnxEpGDmHPKx7Ivpk4nR7THvAevF6AL6q7laWURpO8U420U/1jcZ/lTEkySlLz8zeZghv1j2FNAW1n8AdtADd1zog3RnMH/Sif9qhPtxttOOz3RKq1TNBDkwA3wB9oFhb+ZZfpyuwfP8x8bot0tEvkb7LGPIL+T55fcTS3sdBimatXdSDI9Y1UMKJmWlqsfveh2+DbbVnd40tdlsZuIo7qPt0rS9voNId+OlSwPE7tuTyIOOP27YI+A3st7LLutRTjFSwGngBHA4j7zL4KbPea79++sSkW/IfptnyIfzufG3kyTpd2IXWAfFhzo7UUJD1iYFtgGDGAvR/ULaYXn20uRxDvmldmjaXr9vfHfjd6MDZJ07k9yfcPxpI+ymuAtM0AdpZPUWgBGwNw3/vxjCKzD3Gc+iAFJErm6/zZBtOpvjb+C84nAcuITG15a1q+KYFLjNjB9G95+kuT3MyOQWDvr6jdFK6zeb9+JwL529NyeRh6PZT83Y4+43ob4gZfWcsjsfg2F3P5p7OEx/78c826vRaERkZQ7YLEP+SYxTaYhx0LCbvaiTBlG9JGtTvW82OzyM2ZboZOQ+w67haWCqlYX7KusviDzogQcuDRA7bk8i9xdN2NyPcxu1ZgOdrLJS1ZCG5x1O4LwWQzzCbDzDsyiAFJHVeZ5ZXpr/NGbpqyHQ4zBj8BBwI2ohk7WpspH9wA7HvxOwqdDR15n9fvxdTg6Mrtd+i9dXEOlu/OpCH1n3jsTsYcf3GjzucAvQXc6lE1RWqspAzjh8ZO5/iWavMG8f0bd5QgGkiKxacd0YY2js/ZhhwSya0+VFHTWAEh2ydgHoM9htThrN+nEi37r8Du5j67HOWj9B5EEPvDjcQ2fvrYmx22N8wsyedNhFcWLqpJSVqrrxWQAmgPMGbwcLR2LKSdL+y+3UhYKIrEPvbbrMA+MnQmadERsw6HL8bpY+ty+yUtX7JQW2O/Sa0xXMx2Jn5xS/Gz2F+7pLfqyfIHLXcA+dvbcl0R507AkzHne4lyKAVGfisloZcNGd983sqFn8W0byAZd6LvJzBZAi8k8qxtgey7LJd9OEEPF5M+a8aNrehOotWZuqQ/J+x3c64ekQfDZad+BXFz6gSIysG63/5I+7MfRlP9a7I8ntQYdnzP27DndTBJApuqOTlakykBGYMfdzZva6Wf7nfCEeZ7ZnhP9k3XeZICJtYr/l3N4/mnXwbsD+7NhrFF2ITVNch9ZV1kjaRtEhOWw290c8xn8J0X9AX+89vOy9rS5cvdYHkS8O9zLXdXOScp8HHjfYWw4rtY1aAKkgUlYqB8aB027huJkfzYnvkQ+e54DNrremABFpc/ssg76RLNp7FuNfcf5m+EnU/Y/88zoxbjazRx3/fsjjk0xP3sV/jQy0umCV1jZn/9G7GLu8PXHb7dH2urEXuAfYQu3hZAWQslIONgd+BuxN8/hqHu0EC5u/4IBGoxGRb8h+yznkI/kX00cTMne3OcNxt/sxAushYSPtpr4PyRsNHgbmQsRiV89fedk/5jmbbm0RW/nGHvJOLk5vTa3jXid814wfGDyCOhOX1XOKO/558FHgBB5fy0NylM6Fc/yMuRaXT0Q2un2WMd47knvyhsFfHd7DuAjMUVyf1Aoia1El1G7D/CmHfwsWn2Zy4g7+6F3roXDX3pAnzEwMkHJbjH6fw0PlM5BbqGVHFUDKSpTPQNos+HDRF6S/FZ338PwMbJlUE7aIXBM/twUm+kbzEI+b8yfgL8BHFMOtiqxVADrBbjLYA/bDxPguo5O7GPLOVhasNc3Zs8PdpD1bE/LbHbvH8J04mzFlIGXVyiDSLzn2oRlvmvm75B2fcUvPZfbpk9gicg393BZ42c/n05OHk9xxI4MwAF49x6b6TdaqE7gR4wl3syTxJJ+byHnZz7SqafvaZyLdje6+XsxucLcdDjuBbRhdqJNWWZ1I0ZXPmMGnhr1lcCzPOElnz8V2GLxeRDag52ya3v7PcrMjuB12/B/AGDCPPmwja1N9YrsT2InZU+62L0nCU0xP7uS3X/S1olCtac6eT7qSkGzGuTHAVqALBY+yehkwasZHDscMfyPPOcH8/DDPow/SiEjrPMsMc9m5aP6WOX9xOIYzjIJI+ed1EP0m4DH3+OOQ+zOk/btaEUhe++bsX2Dcawkx68LogdAJbtT69/sq1yLYXMkzdAp6W8Prvs+Y+xl3e8uCHcnNjxPnz3H6hik9BykiLWXmHPRxHhg/GXLrik50A2v+7L/qE1kpK//vBm7F7Ds4SeJxPk+6Jznon17LcbZbEUQ6/5PP5m6jweys4x/jpMAN5UHpqJvbqZ1c9U3dBlZkUd1rJ581ORG9DDquCCq86pi6mcbp9eWoba25crotf1Fwt7Ksq7lwfMX21nQR8obvK912s/mbTVtJmaq/a07jTYTj2JK7dqPIPs5jNmX45252zKIdyY13yObPcWrr5HodqF5ErjMvWMT9UvbbieNph4VodBn0euRWjE1AL0vrPJGVMoqm7c1ubIewjWC9/IjACytKyH0trn0QaeYM+TgLs58Fy0J05glcdGcHMBiwXsAwj2UQYbgFJyZl4FgGX178XMXkLH63Jb9bNWqANwZM3uSrfl1e/u+YO26Ngd/Vgjors6v183rdHAlFN0ah/N3LYLd5MGcEfMm269Zl1WvlK157RMG8YX3Bl8xn7hC82L43OwZgZngMS5bFY3lcl/Jy2tXKW+1vMdXA0/LYZhRjXVfbr/52C+XyKcXxmnMYNffzbnxkztt54ieZyc7ys63rblxREbnOFdeki9nvJ99LIx0xeO7GbnPuwrgZ6GdJvWX11/Oq3il/X3Lt98V5oD6J0qyeanYtvloywBqmNa8fV8aW+flqGtffbLlmZV2u3MuteyVJmGbzrCRpslyZG2OOZhpfC3XTHCyWMc00MAqMYT5F9IyRa9uVVGs+nf08s/ymezjrmprFbDKJcQTsVsy3usdeLDhueRFIWnlChASPKWZVkLT0jVCEm8U/r3vW04hLTjwWl/biNfcl81j5x1r8PThYsfYqkC2W8Yb1WRFwLQaGjSd2IWIYKWZpFSIX83lOJBLK3+OS5ZK6/W7cj/p9bQxevSh/3T4vzutJVaBieqgFhr5Y9lAEkIQySVgG5CHHvTFDSN3yoTwW1fTqYlcLzGvLdWBuWFggehFEBgJYUi6TFftlnXhMwWbNOG/upy2GTzOyT8jyC/RtVRO2iKxft/RdzM5PvQU+nRhnHHvAsbsCvq283hkeDRw3Algo66PqAxXVjXV5A+/1dRblPLWv4mb+ymuxe6y7kW8MZnxxXUXtVA3dmNdtg2WWadQskLXF8l5ZL1ctfs0SO43LxYbXq/Ja3ToiS/vnrMpRnwCxhq8miZTyOFatn3ix7foYYGnShNoxj9Y0AeOLLW+xbtvVMai+V+WsEk6+uIz7ggXmY2TKjBEzThnxDHk6xslrG0S29jmMIU+YH+0jJtvo7NiS5nEQsy4cJ6k7EHkRHLpbwPPiD5nUZdlyM8IyQVtOXDJvPUujRfcmb2YWtx/rXgtYGeYtLV99GUN5F5lf5dh6HjCS8k3vBHyxnDF1QrZ0WbdwxX5V4pI3+9JjUO1X9b0+QK2f12l+DMosMKHcZ/NYLl8LduvLEBa3V5Q3mhGuCBwbN5KUl8a4uP5qm/XLGSlYQvRZC4xmzgjYBTb3XeYnzCuAFJF1r6rzQsetiSd3eYh34L558aa5du0uApewJICptdDVW1zGiuCxuPGvEg+hYb21AMwXg9Am187Fx8WKJAuhvDbHWoucL9YecUmyopHH+romNE0EVYmZWmJlaZ1UawUspltoCARjqJWpyuR6hJA1SfhcmVm1uv1ZOm/RElckMpIrkk5XUztOteNTLBtxz8ta1cv6r1YnVzFBLKdSFytAjpOB54QwT/RZM7ts7ueykP2Dwc3n+ald08E1WhtEQpGK/186GR3torOza/FkAshw0rKMc2aLPy8naXgOMS9PkqzhD95VF1Q2vrY4fZnAMzW76msrUb9P9WWqypJiS/Z9OXNNtle/zNX2u9myy5W1Oq7LHc/lytm4XOPfpzKP0bnkpqG8Uahb70K5bPCIZzOks7Ns3z6rbnxEpO0MeQ9MDbKQD+BJN2lDTylVEiKxsHhTnmdGkl5Z9+RZdRNv5BjBimRHqAtM4uL1s0xYLFPvNUt+JE0CusXy1b0Wl6kXK8GubCmskjRJXUAFRSDVuL4qQVNLZtRnAsPi+nOMBCdf5rGrqx3DZcteJlSq/Vzu+DWWK5RhY5J6GW774r5W66n+ZkvWUQXvgJUxUTUt4gSPBCJ4Tt4xR7YwyfaByWsdQMJ6CCIb1X9QRq5vv6h7fz5Q/vw8UVlHEdkQ3I2XrtLV3naMDzG2rLCu7l9mvuFy+k3fYFPn5ArX3ayM9cv2Y4u/Xyq/b8GaTq/UH59qvsZ5Gt1b9/qHX3F8q+03K2+9Ztu8t2HaSp5ZfO8r5vlFYzZadaKIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIiIhsQP8PA6tEabQW5wIAAAAASUVORK5CYII=" style="height: 24px; vertical-align: middle; margin-right: 8px;" alt="XPayr" /> Pay with crypto. Non-custodial, on-chain checkout.',
            'pluginId' => $pluginId,
            'afterOrderEnabled' => true,
        ];

        if ($ruleId !== null) {
            $paymentMethodData['availabilityRuleId'] = $ruleId;
        }

        // Upload and attach the logo media
        try {
            $mediaId = $this->getOrCreateMediaId($context);
            if ($mediaId) {
                $paymentMethodData['mediaId'] = $mediaId;
            }
        } catch (\Throwable $e) {
            // Ignore media fetch errors to prevent installation failure
        }

        if ($paymentMethodExists) {
            $paymentMethodData['id'] = $paymentMethodExists;
            $paymentRepository->update([$paymentMethodData], $context);
        } else {
            $paymentRepository->create([$paymentMethodData], $context);
        }

        // Also assign to all Sales Channels so it shows up in checkout
        $this->assignToAllSalesChannels($paymentMethodExists ?? $this->getPaymentMethodId(), $context);
    }

    /**
     * Downloads the XPayr logo and persists it to the media repository.
     */
    private function getOrCreateMediaId(Context $context): ?string
    {
        /** @var EntityRepository $mediaRepository */
        $mediaRepository = $this->container->get('media.repository');

        // Check if our logo media already exists to prevent duplicates
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('fileName', 'xpayr-payment-logo'));
        $existingMedia = $mediaRepository->searchIds($criteria, $context)->firstId();

        if ($existingMedia) {
            return $existingMedia;
        }

        /** @var FileFetcher $fileFetcher */
        $fileFetcher = $this->container->get(FileFetcher::class);
        /** @var FileSaver $fileSaver */
        $fileSaver = $this->container->get(FileSaver::class);

        // Fetch long logo from main site
        $mediaFile = $fileFetcher->fetchFileFromURL(new Request(), 'https://xpayr.com/images/logo.png');

        $mediaId = Uuid::randomHex();
        $mediaRepository->create([
            [
                'id' => $mediaId,
                'private' => false,
            ]
        ], $context);

        $fileSaver->persistFileToMedia(
            $mediaFile,
            'xpayr-payment-logo',
            $mediaId,
            $context
        );

        return $mediaId;
    }

    /**
     * Assigns the payment method to all available sales channels.
     */
    private function assignToAllSalesChannels(?string $paymentMethodId, Context $context): void
    {
        if (!$paymentMethodId) {
            return;
        }

        /** @var EntityRepository $salesChannelRepository */
        $salesChannelRepository = $this->container->get('sales_channel.repository');

        $salesChannels = $salesChannelRepository->searchIds(new Criteria(), $context);

        $updates = [];
        foreach ($salesChannels->getIds() as $salesChannelId) {
            $updates[] = [
                'id' => $salesChannelId,
                'paymentMethods' => [
                    ['id' => $paymentMethodId]
                ]
            ];
        }

        if (!empty($updates)) {
            $salesChannelRepository->update($updates, $context);
        }
    }

    /**
     * Toggle the active state of the XPayr payment method.
     */
    private function setPaymentMethodIsActive(bool $active, Context $context): void
    {
        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');

        $paymentMethodId = $this->getPaymentMethodId();

        if (!$paymentMethodId) {
            return;
        }

        $paymentMethod = [
            'id' => $paymentMethodId,
            'active' => $active,
        ];

        $paymentRepository->update([$paymentMethod], $context);
    }

    /**
     * Look up the existing XPayr payment method ID.
     */
    private function getPaymentMethodId(): ?string
    {
        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');

        $criteria = (new Criteria())->addFilter(
            new EqualsFilter('handlerIdentifier', XPayrPayment::class)
        );

        $paymentIds = $paymentRepository->searchIds($criteria, Context::createDefaultContext());

        if ($paymentIds->getTotal() === 0) {
            return null;
        }

        return $paymentIds->getIds()[0];
    }
}
